/**
 * MisterTranslate - Admin JS v4.2.0
 * 
 * Features:
 * - WooCommerce support
 * - Multiple language selection
 * - Cost estimation
 * - Content preview
 * - Draft/Publish modes
 * - Glossary management
 * - Outdated translation detection
 */
(function($) {
    'use strict';
    
    var selectedItems = [];
    var selectedLangs = [];
    var currentItemType = 'page';
    var searchTimeout = null;
    var currentTokensAvailable = mtpData.tokensAvailable || 0;
    var ratios = mtpData.ratios || {};
    
    $(document).ready(function() {
        loadBalance();
        initTabs();
        initSearch();
        initLanguageSelection();
        initTranslation();
        initModals();
        initGlossary();
        initOutdated();
        initPreview();
        
        $('#mtp-refresh-balance').on('click', loadBalance);
    });
    
    // ============================================
    // TABS
    // ============================================
    
    function initTabs() {
        $('.mtp-tab').on('click', function() {
            var tab = $(this).data('tab');
            
            $('.mtp-tab').removeClass('active');
            $(this).addClass('active');
            
            $('.mtp-tab-content').removeClass('active');
            $('#tab-' + tab).addClass('active');
            
            if (tab === 'glossary' || tab === 'outdated') {
                $('#mtp-selection-summary, #mtp-cost-estimate, #mtp-languages-section, .mtp-multilingual-section, .mtp-translate-section').hide();
            } else {
                $('#mtp-languages-section, .mtp-translate-section').show();
                if ($('.mtp-multilingual-section').length) {
                    $('.mtp-multilingual-section').show();
                }
            }
            
            switch (tab) {
                case 'products':
                    currentItemType = 'product';
                    break;
                case 'categories':
                    currentItemType = 'product_cat';
                    break;
                case 'attributes':
                    currentItemType = 'attribute';
                    break;
                case 'glossary':
                case 'outdated':
                    return;
                default:
                    currentItemType = 'page';
            }
            
            clearAllSelections();
        });
    }
    
    // ============================================
    // BALANCE
    // ============================================
    
    function loadBalance() {
        $('#mtp-words-min, #mtp-words-max, #mtp-tokens-exact').text('...');
        
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_check_balance',
                nonce: mtpData.nonce
            },
            success: function(response) {
                if (response.success) {
                    currentTokensAvailable = response.data.tokens_remaining;
                    updateBalanceDisplay(response.data.tokens_remaining);
                    checkZeroBalance(response.data.tokens_remaining);
                } else {
                    $('#mtp-tokens-exact').text('Error');
                }
            }
        });
    }
    
    function updateBalanceDisplay(tokens) {
        var wordsMin = Math.floor(tokens / 5.5);
        var wordsMax = Math.floor(tokens / 4.0);
        
        $('#mtp-words-min').text(numberFormat(wordsMin));
        $('#mtp-words-max').text(numberFormat(wordsMax));
        $('#mtp-tokens-exact').text(numberFormat(tokens));
    }
    
    function checkZeroBalance(tokens) {
        if (tokens <= 0) {
            $('#mtp-zero-balance-alert').show();
        } else {
            $('#mtp-zero-balance-alert').hide();
        }
    }
    
    // ============================================
    // LANGUAGE SELECTION (Multiple)
    // ============================================
    
    function initLanguageSelection() {
        $('#mtp-language-checkboxes input[type="checkbox"]').on('change', function() {
            updateSelectedLanguages();
            updateCostEstimate();
            updateTranslateButton();
        });
        
        $('#mtp-select-all-langs').on('click', function(e) {
            e.preventDefault();
            $('#mtp-language-checkboxes input[type="checkbox"]').prop('checked', true);
            updateSelectedLanguages();
            updateCostEstimate();
            updateTranslateButton();
        });
        
        $('#mtp-clear-all-langs').on('click', function(e) {
            e.preventDefault();
            $('#mtp-language-checkboxes input[type="checkbox"]').prop('checked', false);
            updateSelectedLanguages();
            updateCostEstimate();
            updateTranslateButton();
        });
    }
    
    function updateSelectedLanguages() {
        selectedLangs = [];
        $('#mtp-language-checkboxes input[type="checkbox"]:checked').each(function() {
            selectedLangs.push($(this).val());
        });
    }
    
    // ============================================
    // COST ESTIMATION
    // ============================================
    
    function updateCostEstimate() {
        if (selectedItems.length === 0 || selectedLangs.length === 0) {
            $('#mtp-cost-estimate').hide();
            return;
        }
        
        var itemIds = selectedItems.map(function(item) { return item.id; });
        
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_estimate_cost',
                nonce: mtpData.nonce,
                item_ids: itemIds,
                item_type: currentItemType,
                target_langs: selectedLangs
            },
            success: function(response) {
                if (response.success) {
                    $('#mtp-estimate-words').text(numberFormat(response.data.total_words));
                    $('#mtp-estimate-tokens-min').text(numberFormat(response.data.tokens_min));
                    $('#mtp-estimate-tokens-max').text(numberFormat(response.data.tokens_max));
                    $('#mtp-cost-estimate').show();
                }
            }
        });
    }
    
    // ============================================
    // SEARCH & SELECTION
    // ============================================
    
    function initSearch() {
        $('input[id$="-search"]').on('input', function() {
            var $input = $(this);
            var query = $input.val();
            var type = $input.data('type');
            
            if (searchTimeout) clearTimeout(searchTimeout);
            
            if (query.length < 2) {
                return;
            }
            
            $input.next('.spinner').addClass('is-active');
            searchTimeout = setTimeout(function() {
                searchItems(query, type);
            }, 300);
        });
        
        $('.mtp-load-all').on('click', function() {
            var type = $(this).data('type');
            loadAllItems(type);
        });
        
        $(document).on('change', '.mtp-item-checkbox input', function() {
            var $item = $(this).closest('.mtp-item');
            var itemId = $item.data('id');
            var title = $item.data('title');
            var itemType = $item.data('item-type') || currentItemType;
            
            if ($(this).is(':checked')) {
                if (!isItemSelected(itemId)) {
                    selectedItems.push({ id: itemId, title: title, type: itemType });
                }
            } else {
                selectedItems = selectedItems.filter(function(item) { return item.id !== itemId; });
            }
            
            updateSelectionSummary();
            updateCostEstimate();
            updateTranslateButton();
        });
        
        $('#mtp-clear-all').on('click', function(e) {
            e.preventDefault();
            clearAllSelections();
        });
    }
    
    function searchItems(query, type) {
        var listId = getListIdForType(type);
        
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_get_items',
                nonce: mtpData.nonce,
                search: query,
                item_type: type
            },
            success: function(response) {
                $('.spinner').removeClass('is-active');
                renderItemList(response.success ? response.data : [], listId, type);
            },
            error: function() {
                $('.spinner').removeClass('is-active');
            }
        });
    }
    
    function loadAllItems(type) {
        var $btn = $('.mtp-load-all[data-type="' + type + '"]');
        var listId = getListIdForType(type);
        
        $btn.prop('disabled', true).text('Loading...');
        
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_get_items',
                nonce: mtpData.nonce,
                search: '',
                item_type: type
            },
            success: function(response) {
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-page"></span> Load all');
                renderItemList(response.success ? response.data : [], listId, type);
            },
            error: function() {
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-page"></span> Load all');
            }
        });
    }
    
    function getListIdForType(type) {
        switch (type) {
            case 'product': return '#mtp-product-list';
            case 'product_cat': return '#mtp-category-list';
            case 'attribute': return '#mtp-attribute-list';
            default: return '#mtp-page-list';
        }
    }
    
    function renderItemList(items, listId, type) {
        var $list = $(listId);
        
        if (!items || items.length === 0) {
            $list.html('<p class="mtp-hint">No items found</p>');
            return;
        }
        
        var html = '';
        for (var i = 0; i < items.length; i++) {
            var item = items[i];
            var isSelected = isItemSelected(item.id);
            var checked = isSelected ? ' checked' : '';
            var builder = item.builder ? '<span class="mtp-builder-badge mtp-builder-' + item.builder + '">' + item.builder + '</span>' : '';
            
            html += '<div class="mtp-item" data-id="' + item.id + '" data-title="' + escapeHtml(item.title) + '" data-item-type="' + type + '">' +
                    '<div class="mtp-item-checkbox"><input type="checkbox"' + checked + '></div>' +
                    '<div class="mtp-item-title">' + escapeHtml(item.title) + '</div>' +
                    '<div class="mtp-item-builder">' + builder + '</div>' +
                    '<div class="mtp-item-type"><span>' + (item.type || type) + '</span></div>' +
                    '</div>';
        }
        
        $list.html(html);
        $('#mtp-selection-summary').show();
    }
    
    function isItemSelected(itemId) {
        for (var i = 0; i < selectedItems.length; i++) {
            if (selectedItems[i].id === itemId) return true;
        }
        return false;
    }
    
    function updateSelectionSummary() {
        var $summary = $('#mtp-selection-summary');
        var $count = $('#mtp-selected-count');
        
        if (selectedItems.length > 0) {
            $count.text(selectedItems.length);
            $summary.show();
        } else {
            $summary.hide();
        }
    }
    
    function clearAllSelections() {
        selectedItems = [];
        $('.mtp-item-checkbox input').prop('checked', false);
        updateSelectionSummary();
        updateCostEstimate();
        updateTranslateButton();
    }
    
    function updateTranslateButton() {
        var hasItems = selectedItems.length > 0;
        var hasLangs = selectedLangs.length > 0;
        $('#mtp-translate-btn, #mtp-translate-draft-btn').prop('disabled', !(hasItems && hasLangs));
    }
    
    // ============================================
    // PREVIEW
    // ============================================
    
    function initPreview() {
        $('#mtp-preview-btn').on('click', function(e) {
            e.preventDefault();
            if (selectedItems.length === 0) return;
            
            var item = selectedItems[0];
            
            $.ajax({
                url: mtpData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mtp_preview_content',
                    nonce: mtpData.nonce,
                    item_id: item.id,
                    item_type: item.type || currentItemType
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        $('#mtp-preview-title').text(data.title);
                        $('#mtp-preview-words').text(numberFormat(data.words));
                        $('#mtp-preview-lang').text(data.source_lang.name || 'Unknown');
                        $('#mtp-preview-confidence').text(data.source_lang.confidence ? ' (' + Math.round(data.source_lang.confidence * 100) + '% confidence)' : '');
                        $('#mtp-preview-content-text').text(data.content || 'No content preview available');
                        $('#mtp-preview-modal').show();
                    }
                }
            });
        });
        
        $('#mtp-close-preview').on('click', function() {
            $('#mtp-preview-modal').hide();
        });
    }
    
    // ============================================
    // TRANSLATION
    // ============================================
    
    function initTranslation() {
        $('#mtp-translate-btn').on('click', function() {
            startTranslation(false);
        });
        
        $('#mtp-translate-draft-btn').on('click', function() {
            startTranslation(true);
        });
    }
    
    function startTranslation(asDraft) {
        var linkLang = $('#mtp-link-lang').val() || '';
        var totalTasks = selectedItems.length * selectedLangs.length;
        
        if (totalTasks === 0) return;
        
        $('#mtp-progress-fill').css('width', '0%');
        $('#mtp-progress-status').text('Preparing...');
        $('#mtp-progress-title').text('Translating ' + selectedItems.length + ' item(s) to ' + selectedLangs.length + ' language(s)...');
        $('#mtp-progress-modal').show();
        
        var results = [];
        var tokensTotal = 0;
        var currentTask = 0;
        
        function translateNext() {
            if (currentTask >= totalTasks) {
                finishTranslation(results, tokensTotal, asDraft);
                return;
            }
            
            var itemIndex = Math.floor(currentTask / selectedLangs.length);
            var langIndex = currentTask % selectedLangs.length;
            var item = selectedItems[itemIndex];
            var targetLang = selectedLangs[langIndex];
            var langLabel = mtpData.languages[targetLang] || targetLang;
            
            var percent = Math.round((currentTask / totalTasks) * 100);
            $('#mtp-progress-fill').css('width', percent + '%');
            $('#mtp-progress-status').text('Translating: ' + item.title);
            $('#mtp-progress-lang').text('→ ' + langLabel);
            
            var action = 'mtp_translate_page';
            var dataKey = 'page_id';
            
            if (item.type === 'product') {
                action = 'mtp_translate_product';
                dataKey = 'product_id';
            } else if (item.type === 'product_cat' || item.type === 'product_tag') {
                action = 'mtp_translate_category';
                dataKey = 'term_id';
            }
            
            var postData = {
                action: action,
                nonce: mtpData.nonce,
                target_lang: targetLang,
                link_lang: linkLang,
                as_draft: asDraft ? 'true' : 'false'
            };
            postData[dataKey] = item.id;
            
            $.ajax({
                url: mtpData.ajaxUrl,
                type: 'POST',
                timeout: 300000,
                data: postData,
                success: function(response) {
                    if (response.success) {
                        results.push({ ok: true, item: item, lang: targetLang, data: response.data });
                        tokensTotal += (response.data.tokens_used || 0);
                        
                        if (response.data.new_balance) {
                            currentTokensAvailable = response.data.new_balance.tokens;
                            updateBalanceDisplay(currentTokensAvailable);
                        }
                    } else {
                        results.push({ ok: false, item: item, lang: targetLang, error: response.data });
                    }
                    
                    currentTask++;
                    translateNext();
                },
                error: function(xhr, status, error) {
                    results.push({ ok: false, item: item, lang: targetLang, error: 'Error: ' + error });
                    currentTask++;
                    translateNext();
                }
            });
        }
        
        translateNext();
    }
    
    function finishTranslation(results, tokensTotal, asDraft) {
        $('#mtp-progress-fill').css('width', '100%');
        $('#mtp-progress-status').text('Completed!');
        
        setTimeout(function() {
            $('#mtp-progress-modal').hide();
            showBatchResult(results, tokensTotal, asDraft);
            clearAllSelections();
        }, 500);
    }
    
    function showBatchResult(results, tokensTotal, asDraft) {
        var okCount = 0, failCount = 0, lastBalance = null;
        
        for (var i = 0; i < results.length; i++) {
            if (results[i].ok) {
                okCount++;
                if (results[i].data && results[i].data.new_balance) {
                    lastBalance = results[i].data.new_balance;
                }
            } else {
                failCount++;
            }
        }
        
        var statusText = asDraft ? 'as drafts' : 'and published';
        
        var html = '<div class="mtp-result-hero">' +
                   '<h3>Translation completed</h3>' +
                   '<p>' + okCount + ' successful' + (failCount ? ' · ' + failCount + ' failed' : '') + ' · Created ' + statusText + '</p>' +
                   '</div>';
        
        html += '<div class="mtp-result-stats">' +
                '<div class="mtp-result-stat"><span class="mtp-result-stat-value">' + numberFormat(tokensTotal) + '</span><span class="mtp-result-stat-label">Tokens used</span></div>' +
                '</div>';
        
        if (lastBalance) {
            html += '<div class="mtp-result-balance">' +
                    '<strong>New balance:</strong> ' + numberFormat(lastBalance.words_min) + ' - ' + numberFormat(lastBalance.words_max) + ' words' +
                    '</div>';
        }
        
        html += '<div class="mtp-result-list">';
        for (var j = 0; j < results.length; j++) {
            var r = results[j];
            var langLabel = mtpData.languages[r.lang] || r.lang;
            if (r.ok) {
                html += '<div class="mtp-result-item mtp-result-ok">' +
                        '<span class="dashicons dashicons-yes"></span> ' +
                        escapeHtml(r.item.title) + ' → ' + langLabel + ' ' +
                        '<a href="' + r.data.edit_url + '" target="_blank">Edit</a> · ' +
                        '<a href="' + r.data.view_url + '" target="_blank">View</a>' +
                        '</div>';
            } else {
                html += '<div class="mtp-result-item mtp-result-fail">' +
                        '<span class="dashicons dashicons-no"></span> ' +
                        escapeHtml(r.item.title) + ' → ' + langLabel + ': ' + escapeHtml(r.error) +
                        '</div>';
            }
        }
        html += '</div>';
        
        $('#mtp-result-content').html(html);
        $('#mtp-result-modal').show();
    }
    
    // ============================================
    // GLOSSARY
    // ============================================
    
    function initGlossary() {
        loadGlossary();
        
        $('#mtp-add-never').on('click', function() {
            var term = $('#mtp-never-term').val().trim();
            if (!term) return;
            
            saveGlossaryEntry('add_never', term, '', '');
            $('#mtp-never-term').val('');
        });
        
        $('#mtp-add-specific').on('click', function() {
            var term = $('#mtp-specific-term').val().trim();
            var translation = $('#mtp-specific-translation').val().trim();
            if (!term || !translation) return;
            
            saveGlossaryEntry('add_specific', term, translation, 'all');
            $('#mtp-specific-term, #mtp-specific-translation').val('');
        });
        
        $(document).on('click', '.mtp-glossary-remove', function() {
            var $item = $(this).closest('.mtp-glossary-item');
            var term = $item.data('term');
            var type = $item.data('type');
            
            saveGlossaryEntry(type === 'never' ? 'remove_never' : 'remove_specific', term, '', '');
        });
    }
    
    function loadGlossary() {
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_get_glossary',
                nonce: mtpData.nonce
            },
            success: function(response) {
                if (response.success) {
                    renderGlossary(response.data.glossary);
                }
            }
        });
    }
    
    function saveGlossaryEntry(action, term, translation, lang) {
        $.ajax({
            url: mtpData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'mtp_save_glossary',
                nonce: mtpData.nonce,
                glossary_action: action,
                term: term,
                translation: translation,
                lang: lang
            },
            success: function(response) {
                if (response.success) {
                    renderGlossary(response.data.glossary);
                }
            }
        });
    }
    
    function renderGlossary(glossary) {
        var $neverList = $('#mtp-never-list');
        var $specificList = $('#mtp-specific-list');
        
        // Never translate
        if (glossary.never_translate && glossary.never_translate.length > 0) {
            var html = '';
            for (var i = 0; i < glossary.never_translate.length; i++) {
                var term = glossary.never_translate[i];
                html += '<div class="mtp-glossary-item" data-term="' + escapeHtml(term) + '" data-type="never">' +
                        '<span class="mtp-glossary-term">' + escapeHtml(term) + '</span>' +
                        '<button type="button" class="button-link mtp-glossary-remove"><span class="dashicons dashicons-no"></span></button>' +
                        '</div>';
            }
            $neverList.html(html);
        } else {
            $neverList.html('<p class="mtp-hint">No terms added yet</p>');
        }
        
        // Specific translations
        if (glossary.specific_translations && Object.keys(glossary.specific_translations).length > 0) {
            var html = '';
            for (var term in glossary.specific_translations) {
                var translations = glossary.specific_translations[term];
                var transText = translations.all || Object.values(translations)[0] || '';
                html += '<div class="mtp-glossary-item" data-term="' + escapeHtml(term) + '" data-type="specific">' +
                        '<span class="mtp-glossary-term">' + escapeHtml(term) + '</span>' +
                        '<span class="mtp-glossary-arrow">→</span>' +
                        '<span class="mtp-glossary-translation">' + escapeHtml(transText) + '</span>' +
                        '<button type="button" class="button-link mtp-glossary-remove"><span class="dashicons dashicons-no"></span></button>' +
                        '</div>';
            }
            $specificList.html(html);
        } else {
            $specificList.html('<p class="mtp-hint">No translations added yet</p>');
        }
    }
    
    // ============================================
    // OUTDATED TRANSLATIONS
    // ============================================
    
    function initOutdated() {
        $('#mtp-check-outdated').on('click', function() {
            var $btn = $(this);
            var $list = $('#mtp-outdated-list');
            
            $btn.prop('disabled', true).text('Checking...');
            
            $.ajax({
                url: mtpData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'mtp_check_outdated',
                    nonce: mtpData.nonce
                },
                success: function(response) {
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-search"></span> Check for updates');
                    
                    if (response.success && response.data.items.length > 0) {
                        var html = '<p class="mtp-outdated-count">' + response.data.count + ' translation(s) may need updating:</p>';
                        for (var i = 0; i < response.data.items.length; i++) {
                            var item = response.data.items[i];
                            html += '<div class="mtp-outdated-item">' +
                                    '<div class="mtp-outdated-info">' +
                                    '<strong>' + escapeHtml(item.title) + '</strong> (' + item.lang + ')' +
                                    '<br><small>Source: ' + escapeHtml(item.source_title) + '</small>' +
                                    '</div>' +
                                    '<button type="button" class="button mtp-retranslate-btn" data-id="' + item.id + '">Re-translate</button>' +
                                    '</div>';
                        }
                        $list.html(html);
                    } else {
                        $list.html('<p class="mtp-hint mtp-all-good"><span class="dashicons dashicons-yes"></span> All translations are up to date!</p>');
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-search"></span> Check for updates');
                    $list.html('<p class="mtp-hint">Error checking translations</p>');
                }
            });
        });
        
        $(document).on('click', '.mtp-retranslate-btn', function() {
            var $btn = $(this);
            var translatedId = $btn.data('id');
            
            $btn.prop('disabled', true).text('Updating...');
            
            $.ajax({
                url: mtpData.ajaxUrl,
                type: 'POST',
                timeout: 300000,
                data: {
                    action: 'mtp_retranslate',
                    nonce: mtpData.nonce,
                    translated_id: translatedId
                },
                success: function(response) {
                    if (response.success) {
                        $btn.closest('.mtp-outdated-item').fadeOut(function() {
                            $(this).remove();
                        });
                        loadBalance();
                    } else {
                        $btn.prop('disabled', false).text('Re-translate');
                        alert('Error: ' + response.data);
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).text('Re-translate');
                    alert('Network error');
                }
            });
        });
    }
    
    // ============================================
    // MODALS
    // ============================================
    
    function initModals() {
        $('#mtp-close-result').on('click', function() {
            $('#mtp-result-modal').hide();
        });
        
        $('.mtp-modal').on('click', function(e) {
            if (e.target === this) {
                $(this).hide();
            }
        });
    }
    
    // ============================================
    // UTILS
    // ============================================
    
    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(text));
        return div.innerHTML;
    }
    
    function numberFormat(num) {
        if (num === null || num === undefined) return '0';
        return num.toLocaleString('es-ES');
    }
    
})(jQuery);
