<?php
/**
 * MisterTranslate API Client - Secure Version
 * 
 * Features:
 * - HMAC request signing
 * - Domain verification
 * - Balance caching
 * - Rate limiting integration
 * - Secure credential handling
 * 
 * @package MisterTranslate
 * @since 4.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class MTP_API_Client {
    
    /** @var string API base URL */
    const API_URL = 'https://plugin.misterads.es/translate';
    
    /** @var string */
    private $api_url;
    
    /** @var string */
    private $client_id;
    
    /** @var string */
    private $secret_key;
    
    /** @var int Default timeout in seconds */
    private $timeout = 300;
    
    /** @var int Short timeout for quick operations */
    private $short_timeout = 30;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->api_url = self::API_URL;
        
        // Get credentials securely
        $credentials = MTP_Security::get_credentials();
        $this->client_id = $credentials['client_id'];
        $this->secret_key = $credentials['secret_key'];
    }
    
    /**
     * Check if API is configured
     * 
     * @return bool
     */
    public function is_configured() {
        return !empty($this->client_id) && !empty($this->secret_key);
    }
    
    // =========================================
    // TRANSLATION METHODS
    // =========================================
    
    /**
     * Translate page/post content
     * 
     * @param string $content Content to translate
     * @param string $title Page title
     * @param string $target_lang Target language code
     * @param string $content_type 'standard', 'elementor', or 'avia'
     * @return array Result with success status
     */
    public function translate($content, $title, $target_lang, $content_type = 'standard') {
        // Check rate limit
        $rate_check = MTP_Security::check_rate_limit('translation');
        if (!$rate_check['allowed']) {
            return array('success' => false, 'message' => $rate_check['message']);
        }
        
        if (!$this->is_configured()) {
            return array('success' => false, 'message' => 'API credentials not configured');
        }
        
        // Build glossary instructions
        $glossary_instructions = '';
        if (class_exists('MTP_Glossary')) {
            $glossary_instructions = MTP_Glossary::build_prompt_instructions($target_lang);
        }
        
        $payload = array(
            'action' => 'translate_page',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key,
            'content' => $content,
            'title' => $title,
            'target_lang' => $target_lang,
            'content_type' => $content_type,
            'glossary_instructions' => $glossary_instructions
        );
        
        $result = $this->make_secure_request($payload, $this->timeout);
        
        // Clear balance cache after successful translation
        if ($result['success']) {
            MTP_Security::clear_balance_cache($this->client_id);
        }
        
        return $result;
    }
    
    /**
     * Translate WooCommerce product
     * 
     * @param array $content Product content array
     * @param string $target_lang Target language code
     * @return array Result with success status
     */
    public function translate_product($content, $target_lang) {
        // Check rate limit
        $rate_check = MTP_Security::check_rate_limit('translation');
        if (!$rate_check['allowed']) {
            return array('success' => false, 'message' => $rate_check['message']);
        }
        
        if (!$this->is_configured()) {
            return array('success' => false, 'message' => 'API credentials not configured');
        }
        
        // Build glossary instructions
        $glossary_instructions = '';
        if (class_exists('MTP_Glossary')) {
            $glossary_instructions = MTP_Glossary::build_prompt_instructions($target_lang);
        }
        
        $payload = array(
            'action' => 'translate_product',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key,
            'content' => $content,
            'target_lang' => $target_lang,
            'glossary_instructions' => $glossary_instructions
        );
        
        $result = $this->make_secure_request($payload, $this->timeout);
        
        if ($result['success']) {
            MTP_Security::clear_balance_cache($this->client_id);
        }
        
        return $result;
    }
    
    /**
     * Translate taxonomy term (category, tag, attribute)
     * 
     * @param array $content Term content array
     * @param string $target_lang Target language code
     * @return array Result with success status
     */
    public function translate_term($content, $target_lang) {
        // Check rate limit
        $rate_check = MTP_Security::check_rate_limit('translation');
        if (!$rate_check['allowed']) {
            return array('success' => false, 'message' => $rate_check['message']);
        }
        
        if (!$this->is_configured()) {
            return array('success' => false, 'message' => 'API credentials not configured');
        }
        
        $payload = array(
            'action' => 'translate_term',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key,
            'content' => $content,
            'target_lang' => $target_lang
        );
        
        $result = $this->make_secure_request($payload, $this->timeout);
        
        if ($result['success']) {
            MTP_Security::clear_balance_cache($this->client_id);
        }
        
        return $result;
    }
    
    // =========================================
    // BALANCE & VALIDATION
    // =========================================
    
    /**
     * Check token balance
     * 
     * @param bool $force_refresh Skip cache
     * @return array Balance information
     */
    public function check_balance($force_refresh = false) {
        if (!$this->is_configured()) {
            return array(
                'success' => false, 
                'message' => 'API not configured', 
                'tokens_remaining' => 0
            );
        }
        
        // Check cache first
        if (!$force_refresh) {
            $cached = MTP_Security::get_cached_balance($this->client_id);
            if ($cached !== false) {
                $cached['cached'] = true;
                return $cached;
            }
        }
        
        // Check rate limit for balance checks
        $rate_check = MTP_Security::check_rate_limit('balance');
        if (!$rate_check['allowed']) {
            // Return cached value if available
            $cached = MTP_Security::get_cached_balance($this->client_id);
            if ($cached !== false) {
                $cached['cached'] = true;
                $cached['rate_limited'] = true;
                return $cached;
            }
            return array(
                'success' => false, 
                'message' => $rate_check['message'], 
                'tokens_remaining' => 0
            );
        }
        
        $payload = array(
            'action' => 'check_balance',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key
        );
        
        $response = $this->make_secure_request($payload, $this->short_timeout);
        
        if (!$response['success']) {
            return array(
                'success' => false, 
                'message' => isset($response['message']) ? $response['message'] : 'Unknown error', 
                'tokens_remaining' => 0
            );
        }
        
        $result = array(
            'success' => true,
            'tokens_purchased' => isset($response['tokens_purchased']) ? intval($response['tokens_purchased']) : 0,
            'tokens_used' => isset($response['tokens_used']) ? intval($response['tokens_used']) : 0,
            'tokens_remaining' => isset($response['tokens_remaining']) ? intval($response['tokens_remaining']) : 0,
            'cached' => false
        );
        
        // Cache the result
        MTP_Security::cache_balance($this->client_id, $result);
        
        return $result;
    }
    
    /**
     * Validate API credentials
     * 
     * @return array Validation result
     */
    public function validate_credentials() {
        if (!$this->is_configured()) {
            return array('success' => false, 'message' => 'API not configured');
        }
        
        $payload = array(
            'action' => 'validate',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key
        );
        
        $result = $this->make_secure_request($payload, $this->short_timeout);
        
        // If validation succeeds and server asks to register domain
        if ($result['success'] && isset($result['register_domain']) && $result['register_domain']) {
            $this->register_domain();
        }
        
        return array(
            'success' => isset($result['success']) ? $result['success'] : false,
            'message' => isset($result['message']) ? $result['message'] : 'Unknown response'
        );
    }
    
    /**
     * Register current domain with API server
     */
    private function register_domain() {
        $payload = array(
            'action' => 'register_domain',
            'client_id' => $this->client_id,
            'secret_key' => $this->secret_key,
            'domain' => MTP_Security::get_domain(),
            'site_url' => get_site_url(),
            'fingerprint' => MTP_Security::get_site_fingerprint()
        );
        
        // Fire and forget
        $this->make_secure_request($payload, $this->short_timeout);
    }
    
    // =========================================
    // CORE REQUEST METHOD
    // =========================================
    
    /**
     * Make secure API request with HMAC signing
     * 
     * @param array $payload Request data
     * @param int $timeout Request timeout
     * @return array Response data
     */
    private function make_secure_request($payload, $timeout = null) {
        if ($timeout === null) {
            $timeout = $this->timeout;
        }
        
        // Add security metadata
        $payload = MTP_Security::secure_payload($payload);
        
        // Get secure headers with HMAC signature
        $headers = MTP_Security::get_secure_headers($payload);
        
        $response = wp_remote_post($this->api_url . '/api.php', array(
            'body' => json_encode($payload),
            'headers' => $headers,
            'timeout' => $timeout,
            'sslverify' => true
        ));
        
        // Handle WordPress error
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            
            if (function_exists('mtp_log')) {
                mtp_log('API Request Error', MTP_Security::sanitize_for_log(array(
                    'action' => isset($payload['action']) ? $payload['action'] : 'unknown',
                    'error' => $error_message
                )));
            }
            
            return array('success' => false, 'message' => $error_message);
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Log non-200 responses
        if ($http_code !== 200) {
            if (function_exists('mtp_log')) {
                mtp_log('API HTTP Error', array(
                    'http_code' => $http_code,
                    'action' => isset($payload['action']) ? $payload['action'] : 'unknown'
                ));
            }
        }
        
        // Parse JSON response
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array('success' => false, 'message' => 'Invalid response from API');
        }
        
        // Handle security-specific errors from server
        if (isset($data['error_code'])) {
            return $this->handle_security_error($data['error_code'], $data);
        }
        
        // Standard error handling
        if (!isset($data['success']) || !$data['success']) {
            return array(
                'success' => false, 
                'message' => isset($data['message']) ? $data['message'] : 'Unknown error'
            );
        }
        
        // Success - return all data
        return array_merge(array('success' => true), $data);
    }
    
    /**
     * Handle security-specific error codes
     * 
     * @param string $error_code
     * @param array $data Full response data
     * @return array Error response
     */
    private function handle_security_error($error_code, $data) {
        $messages = array(
            'DOMAIN_MISMATCH' => 'This license is not valid for this domain. Please contact support at support@mistertranslate.com',
            'SIGNATURE_INVALID' => 'Request signature verification failed. Please reinstall the plugin or contact support.',
            'TIMESTAMP_EXPIRED' => 'Request expired. Please check your server time is correct.',
            'RATE_LIMITED' => 'Too many requests. Please wait a few minutes before trying again.',
            'CREDENTIALS_INVALID' => 'Invalid API credentials. Please check your Client ID and Secret Key.',
            'ACCOUNT_SUSPENDED' => 'Your account has been suspended. Please contact support.',
            'INSUFFICIENT_BALANCE' => 'Insufficient token balance. Please purchase more tokens.'
        );
        
        $message = isset($messages[$error_code]) 
            ? $messages[$error_code] 
            : (isset($data['message']) ? $data['message'] : 'Security error: ' . $error_code);
        
        return array(
            'success' => false,
            'message' => $message,
            'error_code' => $error_code
        );
    }
}
