<?php
/**
 * MisterTranslate Glossary Class
 * 
 * Manages translation glossary with:
 * - Terms to never translate (brand names, etc)
 * - Terms with specific translations
 * - Auto-detection of format
 * 
 * @package MisterTranslate
 * @since 4.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class MTP_Glossary {
    
    /** @var string Option name for glossary data */
    const OPTION_NAME = 'mtp_glossary';
    
    /** @var array Cached glossary data */
    private static $cache = null;
    
    /**
     * Get all glossary entries
     * 
     * @return array
     */
    public static function get_all() {
        if (self::$cache === null) {
            self::$cache = get_option(self::OPTION_NAME, array(
                'never_translate' => array(),
                'specific_translations' => array()
            ));
        }
        return self::$cache;
    }
    
    /**
     * Get terms that should never be translated
     * 
     * @return array List of terms
     */
    public static function get_never_translate() {
        $glossary = self::get_all();
        return isset($glossary['never_translate']) ? $glossary['never_translate'] : array();
    }
    
    /**
     * Get terms with specific translations
     * 
     * @return array term => translation pairs
     */
    public static function get_specific_translations() {
        $glossary = self::get_all();
        return isset($glossary['specific_translations']) ? $glossary['specific_translations'] : array();
    }
    
    /**
     * Add term to never translate list
     * 
     * @param string $term Term to preserve
     * @return bool Success
     */
    public static function add_never_translate($term) {
        $term = trim($term);
        if (empty($term)) {
            return false;
        }
        
        $glossary = self::get_all();
        
        if (!in_array($term, $glossary['never_translate'])) {
            $glossary['never_translate'][] = $term;
            return self::save($glossary);
        }
        
        return true;
    }
    
    /**
     * Remove term from never translate list
     * 
     * @param string $term Term to remove
     * @return bool Success
     */
    public static function remove_never_translate($term) {
        $glossary = self::get_all();
        
        $key = array_search($term, $glossary['never_translate']);
        if ($key !== false) {
            unset($glossary['never_translate'][$key]);
            $glossary['never_translate'] = array_values($glossary['never_translate']);
            return self::save($glossary);
        }
        
        return true;
    }
    
    /**
     * Add specific translation
     * 
     * @param string $term Original term
     * @param string $translation Specific translation
     * @param string $lang Target language code (optional, for language-specific)
     * @return bool Success
     */
    public static function add_specific_translation($term, $translation, $lang = 'all') {
        $term = trim($term);
        $translation = trim($translation);
        
        if (empty($term) || empty($translation)) {
            return false;
        }
        
        $glossary = self::get_all();
        
        if (!isset($glossary['specific_translations'][$term])) {
            $glossary['specific_translations'][$term] = array();
        }
        
        $glossary['specific_translations'][$term][$lang] = $translation;
        
        return self::save($glossary);
    }
    
    /**
     * Remove specific translation
     * 
     * @param string $term Term to remove
     * @param string $lang Language code or 'all'
     * @return bool Success
     */
    public static function remove_specific_translation($term, $lang = null) {
        $glossary = self::get_all();
        
        if (isset($glossary['specific_translations'][$term])) {
            if ($lang === null) {
                unset($glossary['specific_translations'][$term]);
            } else {
                unset($glossary['specific_translations'][$term][$lang]);
                if (empty($glossary['specific_translations'][$term])) {
                    unset($glossary['specific_translations'][$term]);
                }
            }
            return self::save($glossary);
        }
        
        return true;
    }
    
    /**
     * Save glossary data
     * 
     * @param array $glossary Glossary data
     * @return bool Success
     */
    private static function save($glossary) {
        self::$cache = $glossary;
        return update_option(self::OPTION_NAME, $glossary);
    }
    
    /**
     * Build glossary instructions for GPT prompt
     * 
     * @param string $target_lang Target language code
     * @return string Instructions to append to prompt
     */
    public static function build_prompt_instructions($target_lang) {
        $glossary = self::get_all();
        $instructions = array();
        
        // Never translate terms
        $never = self::get_never_translate();
        if (!empty($never)) {
            $instructions[] = "IMPORTANT: Do NOT translate these terms, keep them exactly as they are: " . implode(', ', $never);
        }
        
        // Specific translations
        $specific = self::get_specific_translations();
        if (!empty($specific)) {
            $translation_rules = array();
            foreach ($specific as $term => $translations) {
                // Check for language-specific translation first
                if (isset($translations[$target_lang])) {
                    $translation_rules[] = "\"$term\" → \"{$translations[$target_lang]}\"";
                } elseif (isset($translations['all'])) {
                    $translation_rules[] = "\"$term\" → \"{$translations['all']}\"";
                }
            }
            
            if (!empty($translation_rules)) {
                $instructions[] = "IMPORTANT: Use these specific translations: " . implode(', ', $translation_rules);
            }
        }
        
        if (empty($instructions)) {
            return '';
        }
        
        return "\n\n" . implode("\n", $instructions);
    }
    
    /**
     * Import glossary from array (for bulk import)
     * 
     * @param array $data Import data
     * @return bool Success
     */
    public static function import($data) {
        $glossary = self::get_all();
        
        if (isset($data['never_translate']) && is_array($data['never_translate'])) {
            foreach ($data['never_translate'] as $term) {
                if (!in_array($term, $glossary['never_translate'])) {
                    $glossary['never_translate'][] = trim($term);
                }
            }
        }
        
        if (isset($data['specific_translations']) && is_array($data['specific_translations'])) {
            foreach ($data['specific_translations'] as $term => $translations) {
                if (!isset($glossary['specific_translations'][$term])) {
                    $glossary['specific_translations'][$term] = array();
                }
                if (is_array($translations)) {
                    $glossary['specific_translations'][$term] = array_merge(
                        $glossary['specific_translations'][$term],
                        $translations
                    );
                } else {
                    $glossary['specific_translations'][$term]['all'] = $translations;
                }
            }
        }
        
        return self::save($glossary);
    }
    
    /**
     * Export glossary
     * 
     * @return array Exportable data
     */
    public static function export() {
        return self::get_all();
    }
    
    /**
     * Clear all glossary data
     * 
     * @return bool Success
     */
    public static function clear_all() {
        self::$cache = null;
        return delete_option(self::OPTION_NAME);
    }
    
    /**
     * Get glossary statistics
     * 
     * @return array Stats
     */
    public static function get_stats() {
        $glossary = self::get_all();
        
        return array(
            'never_translate_count' => count($glossary['never_translate']),
            'specific_translations_count' => count($glossary['specific_translations'])
        );
    }
}
