<?php
/**
 * MisterTranslate Notifications Class
 * 
 * Handles:
 * - Low balance email notifications
 * - Notification throttling
 * 
 * @package MisterTranslate
 * @since 4.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class MTP_Notifications {
    
    /** @var int Default low balance threshold in tokens */
    const DEFAULT_LOW_BALANCE_THRESHOLD = 1000;
    
    /** @var string Option name for notification settings */
    const OPTION_SETTINGS = 'mtp_notification_settings';
    
    /** @var string Transient key for throttling */
    const THROTTLE_KEY = 'mtp_low_balance_notified';
    
    /** @var int Throttle period in seconds (24 hours) */
    const THROTTLE_PERIOD = 86400;
    
    /**
     * Get notification settings
     * 
     * @return array Settings
     */
    public static function get_settings() {
        return get_option(self::OPTION_SETTINGS, array(
            'low_balance_enabled' => true,
            'low_balance_threshold' => self::DEFAULT_LOW_BALANCE_THRESHOLD,
            'notification_email' => get_option('admin_email')
        ));
    }
    
    /**
     * Save notification settings
     * 
     * @param array $settings Settings to save
     * @return bool Success
     */
    public static function save_settings($settings) {
        $current = self::get_settings();
        $settings = wp_parse_args($settings, $current);
        
        // Sanitize
        $settings['low_balance_enabled'] = (bool) $settings['low_balance_enabled'];
        $settings['low_balance_threshold'] = absint($settings['low_balance_threshold']);
        $settings['notification_email'] = sanitize_email($settings['notification_email']);
        
        return update_option(self::OPTION_SETTINGS, $settings);
    }
    
    /**
     * Check balance and send notification if needed
     * 
     * @param int $tokens_remaining Current token balance
     * @return bool Whether notification was sent
     */
    public static function check_low_balance($tokens_remaining) {
        $settings = self::get_settings();
        
        // Check if notifications are enabled
        if (!$settings['low_balance_enabled']) {
            return false;
        }
        
        // Check if balance is below threshold
        if ($tokens_remaining >= $settings['low_balance_threshold']) {
            return false;
        }
        
        // Check throttle (don't spam)
        if (get_transient(self::THROTTLE_KEY)) {
            return false;
        }
        
        // Send notification
        $sent = self::send_low_balance_email($tokens_remaining, $settings);
        
        if ($sent) {
            // Set throttle
            set_transient(self::THROTTLE_KEY, time(), self::THROTTLE_PERIOD);
        }
        
        return $sent;
    }
    
    /**
     * Send low balance email
     * 
     * @param int $tokens_remaining Current balance
     * @param array $settings Notification settings
     * @return bool Success
     */
    private static function send_low_balance_email($tokens_remaining, $settings) {
        $to = $settings['notification_email'];
        
        if (empty($to) || !is_email($to)) {
            return false;
        }
        
        $site_name = get_bloginfo('name');
        $site_url = get_site_url();
        
        // Estimate words remaining
        $words_min = floor($tokens_remaining / 5.5);
        $words_max = floor($tokens_remaining / 4.0);
        
        $subject = sprintf(
            '[%s] MisterTranslate: Low balance alert - %s tokens remaining',
            $site_name,
            number_format($tokens_remaining)
        );
        
        $message = sprintf(
            "Hello,\n\n" .
            "This is an automated notification from MisterTranslate.\n\n" .
            "Your translation balance on %s is running low:\n\n" .
            "• Tokens remaining: %s\n" .
            "• Estimated words: %s - %s\n" .
            "• Threshold: %s tokens\n\n" .
            "To continue translating without interruption, please purchase more words at:\n" .
            "https://mistertranslate.com/#precio\n\n" .
            "You can manage your settings at:\n" .
            "%s/wp-admin/admin.php?page=mistertranslate-settings\n\n" .
            "---\n" .
            "This notification is sent once every 24 hours when your balance is low.\n" .
            "You can disable these notifications in the plugin settings.",
            $site_name,
            number_format($tokens_remaining),
            number_format($words_min),
            number_format($words_max),
            number_format($settings['low_balance_threshold']),
            $site_url
        );
        
        $headers = array(
            'Content-Type: text/plain; charset=UTF-8',
            'From: MisterTranslate <noreply@' . parse_url($site_url, PHP_URL_HOST) . '>'
        );
        
        return wp_mail($to, $subject, $message, $headers);
    }
    
    /**
     * Reset throttle (for testing)
     */
    public static function reset_throttle() {
        delete_transient(self::THROTTLE_KEY);
    }
    
    /**
     * Check if notification was recently sent
     * 
     * @return bool|int False if not throttled, timestamp if throttled
     */
    public static function is_throttled() {
        return get_transient(self::THROTTLE_KEY);
    }
}
