<?php
/**
 * MisterTranslate Security Class
 * 
 * Handles:
 * - Credential encryption (AES-256-CBC)
 * - Request signing (HMAC-SHA256)
 * - Rate limiting
 * - Domain fingerprinting
 * - Secure logging
 * 
 * @package MisterTranslate
 * @since 4.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class MTP_Security {
    
    /** @var string Encryption cipher */
    const CIPHER = 'aes-256-cbc';
    
    /** @var string HMAC algorithm */
    const HMAC_ALGO = 'sha256';
    
    /** @var int Timestamp tolerance for replay protection (5 minutes) */
    const TIMESTAMP_TOLERANCE = 300;
    
    /** @var int Rate limit: max translations per window */
    const RATE_LIMIT_TRANSLATIONS = 20;
    
    /** @var int Rate limit: max balance checks per window */
    const RATE_LIMIT_BALANCE = 60;
    
    /** @var int Rate limit window in seconds (5 minutes) */
    const RATE_LIMIT_WINDOW = 300;
    
    /** @var int Balance cache time in seconds (5 minutes) */
    const BALANCE_CACHE_TIME = 300;
    
    // =========================================
    // ENCRYPTION
    // =========================================
    
    /**
     * Get encryption key derived from WordPress salts
     * 
     * @return string Binary encryption key
     */
    private static function get_encryption_key() {
        $salt = '';
        
        if (defined('SECURE_AUTH_KEY') && SECURE_AUTH_KEY !== 'put your unique phrase here') {
            $salt = SECURE_AUTH_KEY;
        } elseif (defined('AUTH_KEY') && AUTH_KEY !== 'put your unique phrase here') {
            $salt = AUTH_KEY;
        } else {
            // Fallback: use site URL (less secure but works)
            $salt = get_site_url() . 'mtp_fallback_salt_v1';
        }
        
        return hash('sha256', $salt . 'mtp_encrypt_v1', true);
    }
    
    /**
     * Get HMAC signing key
     * 
     * @return string Binary HMAC key
     */
    private static function get_hmac_key() {
        $salt = '';
        
        if (defined('NONCE_KEY') && NONCE_KEY !== 'put your unique phrase here') {
            $salt = NONCE_KEY;
        } elseif (defined('AUTH_KEY')) {
            $salt = AUTH_KEY;
        } else {
            $salt = get_site_url();
        }
        
        return hash('sha256', $salt . 'mtp_hmac_v1', true);
    }
    
    /**
     * Encrypt sensitive data
     * 
     * @param string $plaintext Data to encrypt
     * @return string Base64-encoded encrypted data with IV prepended
     */
    public static function encrypt($plaintext) {
        if (empty($plaintext)) {
            return '';
        }
        
        $key = self::get_encryption_key();
        $iv_length = openssl_cipher_iv_length(self::CIPHER);
        $iv = openssl_random_pseudo_bytes($iv_length);
        
        $encrypted = openssl_encrypt(
            $plaintext,
            self::CIPHER,
            $key,
            OPENSSL_RAW_DATA,
            $iv
        );
        
        if ($encrypted === false) {
            return '';
        }
        
        // Prepend IV and encode
        return base64_encode($iv . $encrypted);
    }
    
    /**
     * Decrypt sensitive data
     * 
     * @param string $encrypted Base64-encoded encrypted data
     * @return string Decrypted plaintext or empty string on failure
     */
    public static function decrypt($encrypted) {
        if (empty($encrypted)) {
            return '';
        }
        
        $data = base64_decode($encrypted);
        if ($data === false) {
            return '';
        }
        
        $key = self::get_encryption_key();
        $iv_length = openssl_cipher_iv_length(self::CIPHER);
        
        if (strlen($data) < $iv_length) {
            return '';
        }
        
        $iv = substr($data, 0, $iv_length);
        $ciphertext = substr($data, $iv_length);
        
        $decrypted = openssl_decrypt(
            $ciphertext,
            self::CIPHER,
            $key,
            OPENSSL_RAW_DATA,
            $iv
        );
        
        return $decrypted !== false ? $decrypted : '';
    }
    
    // =========================================
    // CREDENTIAL MANAGEMENT
    // =========================================
    
    /**
     * Save credentials securely
     * 
     * @param string $client_id Client ID (not encrypted, like username)
     * @param string $secret_key Secret key (will be encrypted)
     * @return bool Success
     */
    public static function save_credentials($client_id, $secret_key) {
        // Sanitize client_id
        $client_id = sanitize_text_field($client_id);
        
        // Validate format
        if (!empty($client_id) && strpos($client_id, 'client_') !== 0) {
            return false;
        }
        
        // Save client_id (not sensitive)
        update_option('mtp_client_id', $client_id);
        
        if (!empty($secret_key)) {
            // Encrypt secret_key
            $encrypted = self::encrypt($secret_key);
            update_option('mtp_secret_key_enc', $encrypted);
            
            // Store hash for quick validation
            update_option('mtp_secret_key_hash', hash('sha256', $secret_key));
            
            // Store registration info
            update_option('mtp_registered_domain', self::get_domain());
            update_option('mtp_registered_time', time());
        }
        
        // Remove old unencrypted key if exists (migration)
        delete_option('mtp_secret_key');
        
        // Clear any cached data
        self::clear_all_cache();
        
        return true;
    }
    
    /**
     * Get credentials securely
     * 
     * @return array ['client_id' => string, 'secret_key' => string]
     */
    public static function get_credentials() {
        $client_id = get_option('mtp_client_id', '');
        $secret_key = '';
        
        // Try encrypted first (new format)
        $encrypted = get_option('mtp_secret_key_enc', '');
        if (!empty($encrypted)) {
            $secret_key = self::decrypt($encrypted);
        }
        
        // Fallback to unencrypted (old format - for migration)
        if (empty($secret_key)) {
            $secret_key = get_option('mtp_secret_key', '');
            
            // Auto-migrate if found
            if (!empty($secret_key) && !empty($client_id)) {
                self::save_credentials($client_id, $secret_key);
            }
        }
        
        return array(
            'client_id' => $client_id,
            'secret_key' => $secret_key
        );
    }
    
    /**
     * Check if valid credentials are configured
     * 
     * @return bool
     */
    public static function has_credentials() {
        $creds = self::get_credentials();
        return !empty($creds['client_id']) && !empty($creds['secret_key']);
    }
    
    // =========================================
    // REQUEST SIGNING (HMAC)
    // =========================================
    
    /**
     * Sign a request payload
     * 
     * @param array $payload Request data
     * @return array Signature data to include in request
     */
    public static function sign_request($payload) {
        $timestamp = time();
        // PHP 7+ uses random_bytes, fallback to openssl for PHP 5.6
        if (function_exists('random_bytes')) {
            $nonce = bin2hex(random_bytes(16));
        } else {
            $nonce = bin2hex(openssl_random_pseudo_bytes(16));
        }
        
        // Create signature components
        $sign_data = array(
            'timestamp' => $timestamp,
            'nonce' => $nonce,
            'action' => isset($payload['action']) ? $payload['action'] : '',
            'client_id' => isset($payload['client_id']) ? $payload['client_id'] : '',
            'domain' => self::get_domain()
        );
        
        // Create signature string
        $sign_string = implode('|', $sign_data);
        
        // Generate HMAC signature
        $signature = hash_hmac(self::HMAC_ALGO, $sign_string, self::get_hmac_key());
        
        return array(
            'timestamp' => $timestamp,
            'nonce' => $nonce,
            'signature' => $signature
        );
    }
    
    /**
     * Get secure headers for API request
     * 
     * @param array $payload Request payload
     * @return array HTTP headers
     */
    public static function get_secure_headers($payload) {
        $sig = self::sign_request($payload);
        
        return array(
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'X-MTP-Timestamp' => $sig['timestamp'],
            'X-MTP-Nonce' => $sig['nonce'],
            'X-MTP-Signature' => $sig['signature'],
            'X-MTP-Domain' => self::get_domain(),
            'X-MTP-Fingerprint' => self::get_site_fingerprint(),
            'X-MTP-Version' => defined('MTP_VERSION') ? MTP_VERSION : '4.1.0'
        );
    }
    
    /**
     * Add security metadata to payload
     * 
     * @param array $payload Original payload
     * @return array Payload with security data
     */
    public static function secure_payload($payload) {
        $payload['_security'] = array(
            'domain' => self::get_domain(),
            'fingerprint' => self::get_site_fingerprint(),
            'timestamp' => time(),
            'wp_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'plugin_version' => defined('MTP_VERSION') ? MTP_VERSION : '4.1.0'
        );
        
        return $payload;
    }
    
    // =========================================
    // DOMAIN & FINGERPRINT
    // =========================================
    
    /**
     * Get clean domain from site URL
     * 
     * @return string Domain without protocol or www
     */
    public static function get_domain() {
        $url = get_site_url();
        $parsed = parse_url($url);
        $domain = isset($parsed['host']) ? $parsed['host'] : '';
        
        // Remove www prefix
        $domain = preg_replace('/^www\./', '', $domain);
        
        return strtolower($domain);
    }
    
    /**
     * Generate site fingerprint for verification
     * 
     * @return string SHA256 hash of site characteristics
     */
    public static function get_site_fingerprint() {
        global $wp_version;
        
        $data = array(
            'domain' => self::get_domain(),
            'wp_version' => $wp_version,
            'php_version' => PHP_VERSION,
            'site_url' => get_site_url(),
            'charset' => get_bloginfo('charset'),
            'plugin_version' => defined('MTP_VERSION') ? MTP_VERSION : '4.1.0'
        );
        
        return hash('sha256', json_encode($data));
    }
    
    // =========================================
    // RATE LIMITING
    // =========================================
    
    /**
     * Check rate limit for an action
     * 
     * @param string $action 'translation' or 'balance'
     * @return array ['allowed' => bool, 'remaining' => int, 'message' => string]
     */
    public static function check_rate_limit($action = 'translation') {
        $user_id = get_current_user_id();
        $transient_key = 'mtp_rate_' . $action . '_' . $user_id;
        
        $limit = ($action === 'balance') 
            ? self::RATE_LIMIT_BALANCE 
            : self::RATE_LIMIT_TRANSLATIONS;
        
        $current = get_transient($transient_key);
        
        if ($current === false) {
            // First request in window
            set_transient($transient_key, 1, self::RATE_LIMIT_WINDOW);
            return array(
                'allowed' => true,
                'remaining' => $limit - 1,
                'message' => ''
            );
        }
        
        if (intval($current) >= $limit) {
            return array(
                'allowed' => false,
                'remaining' => 0,
                'message' => sprintf(
                    'Rate limit exceeded. Maximum %d %ss per %d minutes. Please wait.',
                    $limit,
                    $action,
                    self::RATE_LIMIT_WINDOW / 60
                )
            );
        }
        
        // Increment counter
        set_transient($transient_key, intval($current) + 1, self::RATE_LIMIT_WINDOW);
        
        return array(
            'allowed' => true,
            'remaining' => $limit - intval($current) - 1,
            'message' => ''
        );
    }
    
    // =========================================
    // CACHING
    // =========================================
    
    /**
     * Get cached balance
     * 
     * @param string $client_id
     * @return array|false Cached balance data or false
     */
    public static function get_cached_balance($client_id) {
        $cache_key = 'mtp_balance_' . md5($client_id);
        return get_transient($cache_key);
    }
    
    /**
     * Cache balance data
     * 
     * @param string $client_id
     * @param array $balance_data
     */
    public static function cache_balance($client_id, $balance_data) {
        $cache_key = 'mtp_balance_' . md5($client_id);
        set_transient($cache_key, $balance_data, self::BALANCE_CACHE_TIME);
        
        // Also store backup for fallback
        update_option($cache_key . '_backup', $balance_data);
    }
    
    /**
     * Clear balance cache
     * 
     * @param string $client_id
     */
    public static function clear_balance_cache($client_id) {
        $cache_key = 'mtp_balance_' . md5($client_id);
        delete_transient($cache_key);
    }
    
    /**
     * Clear all plugin caches
     */
    public static function clear_all_cache() {
        global $wpdb;
        
        // Clear all MTP transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_mtp_%' 
                OR option_name LIKE '_transient_timeout_mtp_%'"
        );
    }
    
    // =========================================
    // VALIDATION
    // =========================================
    
    /**
     * Validate language code
     * 
     * @param string $lang_code
     * @param array $supported_languages
     * @return bool
     */
    public static function validate_language($lang_code, $supported_languages) {
        if (empty($lang_code)) {
            return false;
        }
        return array_key_exists($lang_code, $supported_languages);
    }
    
    /**
     * Validate AJAX request
     * 
     * @return array ['valid' => bool, 'message' => string]
     */
    public static function verify_ajax_request() {
        // Check nonce
        if (!check_ajax_referer('mtp_nonce', 'nonce', false)) {
            return array('valid' => false, 'message' => 'Invalid security token');
        }
        
        // Check capability
        if (!current_user_can('manage_options')) {
            return array('valid' => false, 'message' => 'Permission denied');
        }
        
        // Check referer
        $referer = wp_get_referer();
        if ($referer) {
            $site_host = parse_url(get_site_url(), PHP_URL_HOST);
            $referer_host = parse_url($referer, PHP_URL_HOST);
            
            if ($site_host !== $referer_host) {
                return array('valid' => false, 'message' => 'Invalid request origin');
            }
        }
        
        return array('valid' => true, 'message' => '');
    }
    
    // =========================================
    // LOGGING
    // =========================================
    
    /**
     * Sanitize data for logging (remove sensitive info)
     * 
     * @param mixed $data
     * @return mixed Sanitized data
     */
    public static function sanitize_for_log($data) {
        $sensitive_keys = array(
            'secret_key', 'secret', 'password', 'passwd', 'pwd',
            'api_key', 'apikey', 'token', 'auth', 'authorization',
            'credit_card', 'card_number', 'cvv', 'ssn'
        );
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $lower_key = strtolower($key);
                
                // Check if key contains sensitive word
                foreach ($sensitive_keys as $sensitive) {
                    if (strpos($lower_key, $sensitive) !== false) {
                        $data[$key] = '[REDACTED]';
                        continue 2;
                    }
                }
                
                // Recurse into arrays
                if (is_array($value)) {
                    $data[$key] = self::sanitize_for_log($value);
                }
            }
        } elseif (is_string($data)) {
            // Check for patterns that look like secrets
            if (preg_match('/^[a-f0-9]{32,}$/i', $data)) {
                return '[REDACTED_HASH]';
            }
        }
        
        return $data;
    }
    
    // =========================================
    // EXTERNAL API PROTECTION
    // =========================================
    
    /**
     * Verify request is from internal WordPress context
     * Blocks external/third-party usage of plugin functions
     * 
     * @return bool True if request is valid internal request
     */
    public static function verify_internal_request() {
        // Must be in WordPress admin context
        if (!is_admin()) {
            return false;
        }
        
        // Must have valid admin capabilities
        if (!current_user_can('manage_options')) {
            return false;
        }
        
        // Must be an AJAX request from WordPress
        if (defined('DOING_AJAX') && DOING_AJAX) {
            // Verify the request originated from this WordPress instance
            $referer = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '';
            $admin_url = admin_url();
            
            // Referer must start with our admin URL
            if (strpos($referer, $admin_url) !== 0 && strpos($referer, site_url()) !== 0) {
                return false;
            }
        }
        
        // Additional check: verify WordPress nonce was validated
        // (this should already be done in AJAX handlers, but double-check)
        
        return true;
    }
    
    /**
     * Block request if not internal
     * Should be called at the start of sensitive functions
     * 
     * @param string $context Context for logging
     * @return void Dies if not internal request
     */
    public static function require_internal_request($context = '') {
        if (!self::verify_internal_request()) {
            // Log attempted external access
            if (function_exists('mtp_log')) {
                mtp_log('SECURITY: External access attempt blocked', array(
                    'context' => $context,
                    'ip' => isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'unknown',
                    'referer' => isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'none',
                    'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'none'
                ));
            }
            
            wp_die(
                'MisterTranslate: Access denied. This function is only available within WordPress admin.',
                'Access Denied',
                array('response' => 403)
            );
        }
    }
    
    /**
     * Check if credentials should be accessible
     * Prevents credential extraction via external calls
     * 
     * @return bool
     */
    public static function can_access_credentials() {
        // Only allow in admin context
        if (!is_admin()) {
            return false;
        }
        
        // Only allow for administrators
        if (!current_user_can('manage_options')) {
            return false;
        }
        
        // Block REST API access to credentials
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return false;
        }
        
        return true;
    }
}
