<?php
/**
 * WooCommerce Handler Class
 * Handles translation of products, categories, tags and attributes
 */

if (!defined('ABSPATH')) {
    exit;
}

class MTP_WooCommerce_Handler {
    
    private $api_client;
    
    // Fields to translate in products
    private $translatable_product_fields = array(
        'post_title',       // Product name
        'post_content',     // Long description
        'post_excerpt',     // Short description
        'post_name'         // Slug
    );
    
    // Meta fields to translate
    private $translatable_meta_fields = array(
        '_purchase_note',           // Note sent to customer after purchase
        '_variation_description',   // Variation-specific description
        '_button_text'              // Button text for external products
    );
    
    // Meta fields to sync (copy, not translate)
    private $sync_meta_fields = array(
        '_price', '_regular_price', '_sale_price',
        '_sku', '_stock', '_stock_status',
        '_weight', '_length', '_width', '_height',
        '_manage_stock', '_backorders', '_sold_individually',
        '_virtual', '_downloadable', '_download_limit', '_download_expiry',
        '_tax_status', '_tax_class',
        '_product_image_gallery', '_thumbnail_id'
    );
    
    public function __construct() {
        $this->api_client = new MTP_API_Client();
    }
    
    /**
     * Translate a WooCommerce product
     * 
     * @param int $product_id Product ID
     * @param string $target_lang Target language code
     * @param string $link_lang Multilingual plugin language to link
     * @param bool $as_draft Create as draft instead of published
     */
    public function translate_product($product_id, $target_lang, $link_lang = '', $as_draft = false) {
        $product = wc_get_product($product_id);
        
        if (!$product) {
            return array('success' => false, 'message' => 'Product not found');
        }
        
        // Check balance
        $balance = $this->api_client->check_balance();
        if (!$balance['success']) {
            return array('success' => false, 'message' => 'Could not verify balance');
        }
        
        // Build content to translate
        $content_to_translate = $this->build_product_content($product);
        
        // Count words
        $words_count = MTP_Word_Estimator::count_words($content_to_translate['combined_text']);
        
        // Estimate cost
        $estimate = MTP_Word_Estimator::estimate_cost($words_count, $target_lang);
        $tokens_available = intval($balance['tokens_remaining']);
        $check = MTP_Word_Estimator::check_balance($tokens_available, $estimate);
        
        if (!$check['allowed']) {
            return array('success' => false, 'message' => $check['message']);
        }
        
        // Send to API for translation
        $result = $this->api_client->translate_product(
            $content_to_translate,
            $target_lang
        );
        
        if (!$result['success']) {
            return array('success' => false, 'message' => $result['message']);
        }
        
        // Create translated product (pass draft status)
        $new_product_id = $this->create_translated_product($product, $result, $target_lang, $link_lang, $as_draft);
        
        if (is_wp_error($new_product_id)) {
            return array('success' => false, 'message' => $new_product_id->get_error_message());
        }
        
        // Get new balance
        $new_balance = $this->api_client->check_balance();
        $new_tokens = $new_balance['success'] ? intval($new_balance['tokens_remaining']) : 0;
        $new_range = MTP_Word_Estimator::tokens_to_words_range($new_tokens, $target_lang);
        
        // Check for low balance notification
        if (class_exists('MTP_Notifications')) {
            MTP_Notifications::check_low_balance($new_tokens);
        }
        
        return array(
            'success' => true,
            'message' => 'Product translated successfully!',
            'new_product_id' => $new_product_id,
            'edit_url' => get_edit_post_link($new_product_id, 'raw'),
            'view_url' => get_permalink($new_product_id),
            'tokens_used' => isset($result['tokens_used']) ? $result['tokens_used'] : 0,
            'words_translated' => $words_count,
            'item_type' => 'product',
            'status' => $as_draft ? 'draft' : 'publish',
            'new_balance' => array(
                'tokens' => $new_tokens,
                'words_min' => $new_range['min'],
                'words_max' => $new_range['max']
            )
        );
    }
    
    /**
     * Build product content for translation
     */
    private function build_product_content($product) {
        $content = array(
            'name' => $product->get_name(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'slug' => $product->get_slug(),
            'purchase_note' => $product->get_purchase_note(),
            'button_text' => $product->get_button_text(),
            'variations' => array()
        );
        
        // If variable product, get variation descriptions
        if ($product->is_type('variable')) {
            $variations = $product->get_children();
            foreach ($variations as $variation_id) {
                $variation = wc_get_product($variation_id);
                if ($variation) {
                    $var_desc = $variation->get_description();
                    if (!empty($var_desc)) {
                        $content['variations'][$variation_id] = array(
                            'description' => $var_desc
                        );
                    }
                }
            }
        }
        
        // Combined text for word counting
        $combined = $content['name'] . ' ' . 
                   $content['description'] . ' ' . 
                   $content['short_description'] . ' ' .
                   $content['purchase_note'] . ' ' .
                   $content['button_text'];
        
        foreach ($content['variations'] as $var) {
            $combined .= ' ' . $var['description'];
        }
        
        $content['combined_text'] = $combined;
        
        return $content;
    }
    
    /**
     * Create translated product
     * 
     * @param WC_Product $original_product Original product object
     * @param array $translation_result Translation result from API
     * @param string $target_lang Target language code
     * @param string $link_lang Multilingual plugin language
     * @param bool $as_draft Create as draft
     */
    private function create_translated_product($original_product, $translation_result, $target_lang, $link_lang = '', $as_draft = false) {
        $original_id = $original_product->get_id();
        $original_post = get_post($original_id);
        
        // Get translated content
        $translated = $translation_result['translated_content'];
        
        // Create new product post
        $new_post_data = array(
            'post_title' => !empty($translated['name']) ? $translated['name'] : $original_product->get_name(),
            'post_name' => !empty($translated['slug']) ? sanitize_title($translated['slug']) : $original_post->post_name . '-' . $target_lang,
            'post_content' => !empty($translated['description']) ? $translated['description'] : '',
            'post_excerpt' => !empty($translated['short_description']) ? $translated['short_description'] : '',
            'post_status' => $as_draft ? 'draft' : 'publish',
            'post_type' => 'product',
            'post_author' => get_current_user_id()
        );
        
        $new_product_id = wp_insert_post($new_post_data);
        
        if (is_wp_error($new_product_id)) {
            return $new_product_id;
        }
        
        // Copy and sync meta data
        $this->copy_product_meta($original_id, $new_product_id, $translated, $target_lang);
        
        // Copy product type term
        $product_type = $original_product->get_type();
        wp_set_object_terms($new_product_id, $product_type, 'product_type');
        
        // Copy categories and tags
        $this->copy_product_taxonomies($original_id, $new_product_id);
        
        // Copy product image
        $thumbnail_id = get_post_thumbnail_id($original_id);
        if ($thumbnail_id) {
            set_post_thumbnail($new_product_id, $thumbnail_id);
        }
        
        // Copy gallery images
        $gallery_ids = $original_product->get_gallery_image_ids();
        if (!empty($gallery_ids)) {
            update_post_meta($new_product_id, '_product_image_gallery', implode(',', $gallery_ids));
        }
        
        // Handle variable products - copy variations
        if ($original_product->is_type('variable')) {
            $this->copy_product_variations($original_id, $new_product_id, $translated, $target_lang);
        }
        
        // Store translation reference
        update_post_meta($new_product_id, '_mtp_source_product', $original_id);
        update_post_meta($new_product_id, '_mtp_target_lang', $target_lang);
        update_post_meta($new_product_id, '_mtp_translated_date', current_time('mysql'));
        
        // Link with multilingual plugin
        if (!empty($link_lang)) {
            $this->link_product_translation($original_id, $new_product_id, $link_lang);
        }
        
        // Clear WooCommerce caches
        wc_delete_product_transients($new_product_id);
        
        return $new_product_id;
    }
    
    /**
     * Copy product meta data
     */
    private function copy_product_meta($source_id, $target_id, $translated, $target_lang) {
        // Sync meta fields (copy without translation)
        foreach ($this->sync_meta_fields as $meta_key) {
            $value = get_post_meta($source_id, $meta_key, true);
            if ($value !== '') {
                update_post_meta($target_id, $meta_key, $value);
            }
        }
        
        // Translated meta fields
        if (!empty($translated['purchase_note'])) {
            update_post_meta($target_id, '_purchase_note', $translated['purchase_note']);
        }
        
        if (!empty($translated['button_text'])) {
            update_post_meta($target_id, '_button_text', $translated['button_text']);
        }
        
        // Copy product attributes
        $attributes = get_post_meta($source_id, '_product_attributes', true);
        if (!empty($attributes)) {
            update_post_meta($target_id, '_product_attributes', $attributes);
        }
        
        // Copy other important meta
        $other_meta = array(
            '_visibility', '_featured', '_product_version',
            '_wp_page_template', '_default_attributes',
            '_crosssell_ids', '_upsell_ids'
        );
        
        foreach ($other_meta as $meta_key) {
            $value = get_post_meta($source_id, $meta_key, true);
            if ($value !== '') {
                update_post_meta($target_id, $meta_key, $value);
            }
        }
    }
    
    /**
     * Copy product taxonomies (categories, tags, attributes)
     */
    private function copy_product_taxonomies($source_id, $target_id) {
        // Product categories
        $cats = wp_get_object_terms($source_id, 'product_cat', array('fields' => 'ids'));
        if (!is_wp_error($cats) && !empty($cats)) {
            wp_set_object_terms($target_id, $cats, 'product_cat');
        }
        
        // Product tags
        $tags = wp_get_object_terms($source_id, 'product_tag', array('fields' => 'ids'));
        if (!is_wp_error($tags) && !empty($tags)) {
            wp_set_object_terms($target_id, $tags, 'product_tag');
        }
        
        // Product attributes (pa_*)
        $attribute_taxonomies = wc_get_attribute_taxonomies();
        foreach ($attribute_taxonomies as $attribute) {
            $taxonomy = 'pa_' . $attribute->attribute_name;
            $terms = wp_get_object_terms($source_id, $taxonomy, array('fields' => 'ids'));
            if (!is_wp_error($terms) && !empty($terms)) {
                wp_set_object_terms($target_id, $terms, $taxonomy);
            }
        }
        
        // Shipping class
        $shipping_class = wp_get_object_terms($source_id, 'product_shipping_class', array('fields' => 'ids'));
        if (!is_wp_error($shipping_class) && !empty($shipping_class)) {
            wp_set_object_terms($target_id, $shipping_class, 'product_shipping_class');
        }
    }
    
    /**
     * Copy product variations
     */
    private function copy_product_variations($source_id, $target_id, $translated, $target_lang) {
        $source_product = wc_get_product($source_id);
        $variation_ids = $source_product->get_children();
        
        $translated_variations = isset($translated['variations']) ? $translated['variations'] : array();
        
        foreach ($variation_ids as $variation_id) {
            $variation = wc_get_product($variation_id);
            if (!$variation) continue;
            
            $variation_post = get_post($variation_id);
            
            // Create new variation
            $new_variation_data = array(
                'post_title' => $variation_post->post_title,
                'post_name' => $variation_post->post_name . '-' . $target_lang,
                'post_status' => 'publish',
                'post_type' => 'product_variation',
                'post_parent' => $target_id,
                'post_author' => get_current_user_id()
            );
            
            $new_variation_id = wp_insert_post($new_variation_data);
            
            if (is_wp_error($new_variation_id)) {
                continue;
            }
            
            // Copy variation meta
            $variation_meta_keys = array(
                '_price', '_regular_price', '_sale_price',
                '_sku', '_stock', '_stock_status',
                '_weight', '_length', '_width', '_height',
                '_manage_stock', '_backorders',
                '_virtual', '_downloadable',
                '_thumbnail_id'
            );
            
            foreach ($variation_meta_keys as $meta_key) {
                $value = get_post_meta($variation_id, $meta_key, true);
                if ($value !== '') {
                    update_post_meta($new_variation_id, $meta_key, $value);
                }
            }
            
            // Copy variation attributes
            $var_attributes = get_post_meta($variation_id);
            foreach ($var_attributes as $key => $values) {
                if (strpos($key, 'attribute_') === 0) {
                    update_post_meta($new_variation_id, $key, $values[0]);
                }
            }
            
            // Set translated description if available
            if (isset($translated_variations[$variation_id]['description'])) {
                update_post_meta($new_variation_id, '_variation_description', $translated_variations[$variation_id]['description']);
            } else {
                $original_desc = get_post_meta($variation_id, '_variation_description', true);
                if (!empty($original_desc)) {
                    update_post_meta($new_variation_id, '_variation_description', $original_desc);
                }
            }
            
            // Store reference
            update_post_meta($new_variation_id, '_mtp_source_variation', $variation_id);
        }
        
        // Update variable product lookup table
        if (function_exists('wc_update_product_lookup_tables')) {
            wc_update_product_lookup_tables($target_id);
        }
    }
    
    /**
     * Translate a taxonomy term (category, tag, attribute term)
     */
    public function translate_term($term_id, $taxonomy, $target_lang, $link_lang = '') {
        $term = get_term($term_id, $taxonomy);
        
        if (!$term || is_wp_error($term)) {
            return array('success' => false, 'message' => 'Term not found');
        }
        
        // Check balance
        $balance = $this->api_client->check_balance();
        if (!$balance['success']) {
            return array('success' => false, 'message' => 'Could not verify balance');
        }
        
        // Build content to translate
        $content = array(
            'name' => $term->name,
            'description' => $term->description,
            'slug' => $term->slug
        );
        
        $combined_text = $term->name . ' ' . $term->description;
        $words_count = MTP_Word_Estimator::count_words($combined_text);
        
        // Estimate cost
        $estimate = MTP_Word_Estimator::estimate_cost($words_count, $target_lang);
        $tokens_available = intval($balance['tokens_remaining']);
        $check = MTP_Word_Estimator::check_balance($tokens_available, $estimate);
        
        if (!$check['allowed']) {
            return array('success' => false, 'message' => $check['message']);
        }
        
        // Translate via API
        $result = $this->api_client->translate_term($content, $target_lang);
        
        if (!$result['success']) {
            return array('success' => false, 'message' => $result['message']);
        }
        
        $translated = $result['translated_content'];
        
        // Create new term
        $new_term_data = array(
            'description' => !empty($translated['description']) ? $translated['description'] : '',
            'slug' => !empty($translated['slug']) ? sanitize_title($translated['slug']) : $term->slug . '-' . $target_lang,
            'parent' => $term->parent
        );
        
        $new_term = wp_insert_term(
            !empty($translated['name']) ? $translated['name'] : $term->name,
            $taxonomy,
            $new_term_data
        );
        
        if (is_wp_error($new_term)) {
            return array('success' => false, 'message' => $new_term->get_error_message());
        }
        
        $new_term_id = $new_term['term_id'];
        
        // Store translation reference
        update_term_meta($new_term_id, '_mtp_source_term', $term_id);
        update_term_meta($new_term_id, '_mtp_target_lang', $target_lang);
        update_term_meta($new_term_id, '_mtp_translated_date', current_time('mysql'));
        
        // Link with multilingual plugin
        if (!empty($link_lang)) {
            $this->link_term_translation($term_id, $new_term_id, $taxonomy, $link_lang);
        }
        
        // Get new balance
        $new_balance = $this->api_client->check_balance();
        $new_tokens = $new_balance['success'] ? intval($new_balance['tokens_remaining']) : 0;
        $new_range = MTP_Word_Estimator::tokens_to_words_range($new_tokens, $target_lang);
        
        return array(
            'success' => true,
            'message' => 'Term translated successfully!',
            'new_term_id' => $new_term_id,
            'edit_url' => get_edit_term_link($new_term_id, $taxonomy),
            'view_url' => get_term_link($new_term_id, $taxonomy),
            'tokens_used' => isset($result['tokens_used']) ? $result['tokens_used'] : 0,
            'words_translated' => $words_count,
            'item_type' => $taxonomy,
            'new_balance' => array(
                'tokens' => $new_tokens,
                'words_min' => $new_range['min'],
                'words_max' => $new_range['max']
            )
        );
    }
    
    /**
     * Link product translation with multilingual plugin
     */
    private function link_product_translation($original_id, $translated_id, $target_lang) {
        // Polylang
        if (function_exists('pll_set_post_language') && function_exists('pll_save_post_translations')) {
            $source_lang = function_exists('pll_get_post_language') 
                ? pll_get_post_language($original_id) 
                : pll_default_language();
            
            pll_set_post_language($translated_id, $target_lang);
            
            $translations = function_exists('pll_get_post_translations') 
                ? pll_get_post_translations($original_id) 
                : array();
            
            if (!empty($source_lang)) {
                $translations[$source_lang] = $original_id;
            }
            $translations[$target_lang] = $translated_id;
            
            pll_save_post_translations($translations);
            
            update_post_meta($translated_id, '_mtp_multilingual_plugin', 'polylang');
            return;
        }
        
        // WPML
        if (has_filter('wpml_element_type') && has_action('wpml_set_element_language_details')) {
            $element_type = apply_filters('wpml_element_type', 'product');
            $trid = apply_filters('wpml_element_trid', null, $original_id, $element_type);
            
            if (!empty($trid)) {
                $source_lang_details = apply_filters('wpml_element_language_details', null, array(
                    'element_id' => $original_id,
                    'element_type' => $element_type
                ));
                $source_lang = isset($source_lang_details->language_code) ? $source_lang_details->language_code : null;
                
                do_action('wpml_set_element_language_details', array(
                    'element_id' => $translated_id,
                    'element_type' => $element_type,
                    'trid' => $trid,
                    'language_code' => $target_lang,
                    'source_language_code' => $source_lang
                ));
                
                update_post_meta($translated_id, '_mtp_multilingual_plugin', 'wpml');
            }
        }
    }
    
    /**
     * Link term translation with multilingual plugin
     */
    private function link_term_translation($original_id, $translated_id, $taxonomy, $target_lang) {
        // Polylang
        if (function_exists('pll_set_term_language') && function_exists('pll_save_term_translations')) {
            $source_lang = function_exists('pll_get_term_language') 
                ? pll_get_term_language($original_id) 
                : pll_default_language();
            
            pll_set_term_language($translated_id, $target_lang);
            
            $translations = function_exists('pll_get_term_translations') 
                ? pll_get_term_translations($original_id) 
                : array();
            
            if (!empty($source_lang)) {
                $translations[$source_lang] = $original_id;
            }
            $translations[$target_lang] = $translated_id;
            
            pll_save_term_translations($translations);
            
            update_term_meta($translated_id, '_mtp_multilingual_plugin', 'polylang');
            return;
        }
        
        // WPML
        if (has_filter('wpml_element_type') && has_action('wpml_set_element_language_details')) {
            $element_type = apply_filters('wpml_element_type', $taxonomy);
            $trid = apply_filters('wpml_element_trid', null, $original_id, 'tax_' . $taxonomy);
            
            if (!empty($trid)) {
                do_action('wpml_set_element_language_details', array(
                    'element_id' => $translated_id,
                    'element_type' => 'tax_' . $taxonomy,
                    'trid' => $trid,
                    'language_code' => $target_lang
                ));
                
                update_term_meta($translated_id, '_mtp_multilingual_plugin', 'wpml');
            }
        }
    }
}
