<?php
/**
 * Plugin Name: MisterTranslate
 * Plugin URI: https://mistertranslate.com
 * Description: Translate pages, posts and WooCommerce products using AI. Supports Elementor, Avia/Enfold and standard WordPress.
 * Version: 4.2.1
 * Author: MisterTranslate
 * Author URI: https://mistertranslate.com
 * Text Domain: mistertranslate
 * Requires at least: 5.0
 * Requires PHP: 5.6
 */

if (!defined('ABSPATH')) {
    exit;
}

define('MTP_VERSION', '4.2.1');
define('MTP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MTP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MTP_DEBUG_LOG', MTP_PLUGIN_DIR . 'debug.log');

// Secure logging function
function mtp_log($message, $data = null) {
    if (!defined('WP_DEBUG') || !WP_DEBUG) return;
    
    $timestamp = date('Y-m-d H:i:s');
    $log = "[$timestamp] $message";
    
    if ($data !== null) {
        // Sanitize sensitive data before logging
        if (class_exists('MTP_Security')) {
            $data = MTP_Security::sanitize_for_log($data);
        }
        $log .= "\n" . print_r($data, true);
    }
    
    $log .= "\n" . str_repeat('-', 50) . "\n";
    file_put_contents(MTP_DEBUG_LOG, $log, FILE_APPEND);
}

// Load classes
require_once MTP_PLUGIN_DIR . 'includes/class-security.php';
require_once MTP_PLUGIN_DIR . 'includes/class-api-client.php';
require_once MTP_PLUGIN_DIR . 'includes/class-word-estimator.php';
require_once MTP_PLUGIN_DIR . 'includes/class-glossary.php';
require_once MTP_PLUGIN_DIR . 'includes/class-notifications.php';
require_once MTP_PLUGIN_DIR . 'includes/class-language-detector.php';

/**
 * Main class
 */
class MisterTranslate {
    
    private static $instance = null;
    private $wc_handler = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Initialize WooCommerce handler if WC is active
        if ($this->is_woocommerce_active()) {
            require_once MTP_PLUGIN_DIR . 'includes/class-woocommerce-handler.php';
            $this->wc_handler = new MTP_WooCommerce_Handler();
        }
        
        // AJAX handlers
        add_action('wp_ajax_mtp_translate_page', array($this, 'ajax_translate_page'));
        add_action('wp_ajax_mtp_translate_product', array($this, 'ajax_translate_product'));
        add_action('wp_ajax_mtp_translate_category', array($this, 'ajax_translate_category'));
        add_action('wp_ajax_mtp_get_items', array($this, 'ajax_get_items'));
        add_action('wp_ajax_mtp_check_balance', array($this, 'ajax_check_balance'));
        
        // New v4.2 AJAX handlers
        add_action('wp_ajax_mtp_estimate_cost', array($this, 'ajax_estimate_cost'));
        add_action('wp_ajax_mtp_preview_content', array($this, 'ajax_preview_content'));
        add_action('wp_ajax_mtp_detect_language', array($this, 'ajax_detect_language'));
        add_action('wp_ajax_mtp_check_outdated', array($this, 'ajax_check_outdated'));
        add_action('wp_ajax_mtp_retranslate', array($this, 'ajax_retranslate'));
        add_action('wp_ajax_mtp_save_glossary', array($this, 'ajax_save_glossary'));
        add_action('wp_ajax_mtp_get_glossary', array($this, 'ajax_get_glossary'));
        
        // Quick action in posts list
        add_filter('post_row_actions', array($this, 'add_translate_row_action'), 10, 2);
        add_filter('page_row_actions', array($this, 'add_translate_row_action'), 10, 2);
        
        // WooCommerce product quick action (using post_row_actions filter)
        // Note: Products are custom post type, already covered by post_row_actions
    }
    
    /**
     * Check if WooCommerce is active
     */
    public function is_woocommerce_active() {
        return class_exists('WooCommerce');
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'MisterTranslate',
            'MisterTranslate',
            'manage_options',
            'mistertranslate',
            array($this, 'render_admin_page'),
            'dashicons-translation',
            30
        );
        
        add_submenu_page(
            'mistertranslate',
            'Settings',
            'Settings',
            'manage_options',
            'mistertranslate-settings',
            array($this, 'render_settings_page')
        );
    }
    
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'mistertranslate') === false) {
            return;
        }
        
        wp_enqueue_style('mtp-admin', MTP_PLUGIN_URL . 'assets/admin.css', array(), MTP_VERSION);
        wp_enqueue_script('mtp-admin', MTP_PLUGIN_URL . 'assets/admin.js', array('jquery'), MTP_VERSION, true);
        
        $api_client = new MTP_API_Client();
        $balance = $api_client->check_balance();
        $tokens = $balance['success'] ? intval($balance['tokens_remaining']) : 0;
        
        wp_localize_script('mtp-admin', 'mtpData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mtp_nonce'),
            'languages' => $this->get_supported_languages(),
            'tokensAvailable' => $tokens,
            'ratios' => MTP_Word_Estimator::get_ratios_for_js(),
            'globalRatioPred' => MTP_Word_Estimator::GLOBAL_RATIO_PRED,
            'globalRatioMax' => MTP_Word_Estimator::GLOBAL_RATIO_MAX,
            'woocommerceActive' => $this->is_woocommerce_active(),
            'i18n' => array(
                'translating' => 'Translating...',
                'completed' => 'Completed!',
                'preparing' => 'Preparing...',
                'error' => 'Error',
                'success' => 'Success',
                'selected' => 'selected',
                'loadAll' => 'Load all',
                'loading' => 'Loading...',
                'noResults' => 'No results found',
                'searchHint' => 'Type at least 2 characters...'
            )
        ));
    }
    
    public function get_supported_languages() {
        return array(
            'en' => 'English', 'es' => 'Español', 'fr' => 'Français',
            'de' => 'Deutsch', 'it' => 'Italiano', 'pt' => 'Português',
            'ca' => 'Català', 'nl' => 'Nederlands', 'ru' => 'Русский',
            'zh' => '中文', 'ja' => '日本語', 'ko' => '한국어',
            'ar' => 'العربية', 'pl' => 'Polski', 'sv' => 'Svenska'
        );
    }
    
    // ==========================================
    // PAGE BUILDER DETECTION
    // ==========================================
    
    private function detect_page_builder($page_id) {
        $elementor_data = get_post_meta($page_id, '_elementor_data', true);
        if (!empty($elementor_data)) {
            $decoded = json_decode($elementor_data, true);
            if (is_array($decoded) && !empty($decoded)) {
                return 'elementor';
            }
        }
        
        $avia_content = get_post_meta($page_id, '_aviaLayoutBuilderCleanData', true);
        if (!empty($avia_content)) {
            return 'avia';
        }
        
        return 'standard';
    }
    
    private function get_page_content($page_id) {
        $post = get_post($page_id);
        if (!$post) {
            return array('content' => '', 'type' => 'standard');
        }
        
        $builder = $this->detect_page_builder($page_id);
        
        switch ($builder) {
            case 'elementor':
                return array(
                    'content' => get_post_meta($page_id, '_elementor_data', true),
                    'type' => 'elementor'
                );
            
            case 'avia':
                return array(
                    'content' => get_post_meta($page_id, '_aviaLayoutBuilderCleanData', true),
                    'type' => 'avia'
                );
            
            default:
                return array(
                    'content' => $post->post_content,
                    'type' => 'standard'
                );
        }
    }
    
    // ==========================================
    // AJAX: TRANSLATE PAGE
    // ==========================================
    
    public function ajax_translate_page() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $page_id = isset($_POST['page_id']) ? intval($_POST['page_id']) : 0;
        $target_lang = isset($_POST['target_lang']) ? sanitize_text_field($_POST['target_lang']) : '';
        $link_lang = isset($_POST['link_lang']) ? sanitize_text_field($_POST['link_lang']) : '';
        $as_draft = isset($_POST['as_draft']) && $_POST['as_draft'] === 'true';
        
        if (!$page_id || !$target_lang) {
            wp_send_json_error('Missing required parameters');
        }
        
        // Validate language
        if (!MTP_Security::validate_language($target_lang, $this->get_supported_languages())) {
            wp_send_json_error('Invalid target language');
        }
        
        $post = get_post($page_id);
        if (!$post) {
            wp_send_json_error('Page not found');
        }
        
        $content_data = $this->get_page_content($page_id);
        $content = $content_data['content'];
        $content_type = $content_data['type'];
        
        if (empty($content)) {
            wp_send_json_error('No content found in this page');
        }
        
        $api_client = new MTP_API_Client();
        $balance = $api_client->check_balance();
        
        if (!$balance['success']) {
            wp_send_json_error('Could not verify balance: ' . $balance['message']);
        }
        
        // Count words
        if ($content_type === 'elementor') {
            $temp_data = json_decode($content, true);
            $temp_text = $this->extract_elementor_text($temp_data);
            $words_count = MTP_Word_Estimator::count_words($temp_text);
        } else {
            $words_count = MTP_Word_Estimator::count_words($content);
        }
        
        $estimate = MTP_Word_Estimator::estimate_cost($words_count, $target_lang);
        $tokens_available = intval($balance['tokens_remaining']);
        $check = MTP_Word_Estimator::check_balance($tokens_available, $estimate);
        
        if (!$check['allowed']) {
            wp_send_json_error($check['message']);
        }
        
        // ELEMENTOR: Process on plugin side (extract text, translate, rebuild JSON)
        if ($content_type === 'elementor') {
            $result = $this->translate_elementor_segments($content, $post->post_title, $target_lang);
        } else {
            $result = $api_client->translate($content, $post->post_title, $target_lang, $content_type);
        }
        
        if (!$result['success']) {
            wp_send_json_error($result['message']);
        }
        
        // Pass draft status to create function
        $new_page_id = $this->create_translated_page($post, $result, $target_lang, $link_lang, $content_type, $as_draft);
        
        if (is_wp_error($new_page_id)) {
            wp_send_json_error('Failed to create page: ' . $new_page_id->get_error_message());
        }
        
        // Store source hash for change detection
        MTP_Translation_Tracker::store_source_hash($new_page_id, $page_id, $content);
        
        $new_balance = $api_client->check_balance();
        $new_tokens = $new_balance['success'] ? intval($new_balance['tokens_remaining']) : 0;
        $new_range = MTP_Word_Estimator::tokens_to_words_range($new_tokens, $target_lang);
        
        // Check for low balance notification
        MTP_Notifications::check_low_balance($new_tokens);
        
        wp_send_json_success(array(
            'message' => 'Translation completed!',
            'new_page_id' => $new_page_id,
            'edit_url' => get_edit_post_link($new_page_id, 'raw'),
            'view_url' => get_permalink($new_page_id),
            'tokens_used' => isset($result['tokens_used']) ? $result['tokens_used'] : 0,
            'words_translated' => $words_count,
            'content_type' => $content_type,
            'item_type' => 'page',
            'status' => $as_draft ? 'draft' : 'publish',
            'new_balance' => array(
                'tokens' => $new_tokens,
                'words_min' => $new_range['min'],
                'words_max' => $new_range['max']
            )
        ));
    }
    
    // ==========================================
    // AJAX: TRANSLATE PRODUCT (WooCommerce)
    // ==========================================
    
    public function ajax_translate_product() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        if (!$this->is_woocommerce_active()) {
            wp_send_json_error('WooCommerce is not active');
        }
        
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $target_lang = isset($_POST['target_lang']) ? sanitize_text_field($_POST['target_lang']) : '';
        $link_lang = isset($_POST['link_lang']) ? sanitize_text_field($_POST['link_lang']) : '';
        $as_draft = isset($_POST['as_draft']) && $_POST['as_draft'] === 'true';
        
        if (!$product_id || !$target_lang) {
            wp_send_json_error('Missing required parameters');
        }
        
        // Validate language
        if (!MTP_Security::validate_language($target_lang, $this->get_supported_languages())) {
            wp_send_json_error('Invalid target language');
        }
        
        $result = $this->wc_handler->translate_product($product_id, $target_lang, $link_lang, $as_draft);
        
        if (!$result['success']) {
            wp_send_json_error($result['message']);
        }
        
        wp_send_json_success($result);
    }
    
    // ==========================================
    // AJAX: TRANSLATE CATEGORY (WooCommerce)
    // ==========================================
    
    public function ajax_translate_category() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        if (!$this->is_woocommerce_active()) {
            wp_send_json_error('WooCommerce is not active');
        }
        
        $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
        $taxonomy = isset($_POST['taxonomy']) ? sanitize_text_field($_POST['taxonomy']) : 'product_cat';
        $target_lang = isset($_POST['target_lang']) ? sanitize_text_field($_POST['target_lang']) : '';
        $link_lang = isset($_POST['link_lang']) ? sanitize_text_field($_POST['link_lang']) : '';
        
        if (!$term_id || !$target_lang) {
            wp_send_json_error('Missing required parameters');
        }
        
        // Validate language
        if (!MTP_Security::validate_language($target_lang, $this->get_supported_languages())) {
            wp_send_json_error('Invalid target language');
        }
        
        // Validate taxonomy
        $allowed_taxonomies = array('product_cat', 'product_tag', 'pa_color', 'pa_size', 'pa_material');
        if (strpos($taxonomy, 'pa_') === 0 || in_array($taxonomy, $allowed_taxonomies)) {
            // Valid WooCommerce taxonomy
        } else {
            wp_send_json_error('Invalid taxonomy');
        }
        
        $result = $this->wc_handler->translate_term($term_id, $taxonomy, $target_lang, $link_lang);
        
        if (!$result['success']) {
            wp_send_json_error($result['message']);
        }
        
        wp_send_json_success($result);
    }
    
    // ==========================================
    // AJAX: GET ITEMS (Pages, Products, Categories)
    // ==========================================
    
    public function ajax_get_items() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $item_type = isset($_POST['item_type']) ? sanitize_text_field($_POST['item_type']) : 'page';
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $search = trim($search);
        
        global $wpdb;
        
        switch ($item_type) {
            case 'product':
                $results = $this->get_products($search);
                break;
            case 'product_cat':
                $results = $this->get_product_categories($search);
                break;
            case 'product_tag':
                $results = $this->get_product_tags($search);
                break;
            case 'attribute':
                $results = $this->get_product_attributes($search);
                break;
            default:
                $results = $this->get_pages($search);
        }
        
        wp_send_json_success($results);
    }
    
    private function get_pages($search = '') {
        global $wpdb;
        
        $where = "p.post_status = 'publish' AND p.post_type IN ('page','post')";
        $params = array();
        
        if (!empty($search) && strlen($search) >= 2) {
            $like = '%' . $wpdb->esc_like($search) . '%';
            $where .= " AND (p.post_title LIKE %s OR p.post_name LIKE %s)";
            $params[] = $like;
            $params[] = $like;
        }
        
        $sql = "SELECT p.ID, p.post_title, p.post_type
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_mtp_source_page'
                WHERE {$where}
                  AND pm1.meta_id IS NULL
                ORDER BY p.post_title ASC
                LIMIT 200";
        
        if (!empty($params)) {
            $sql = $wpdb->prepare($sql, $params);
        }
        
        $pages = $wpdb->get_results($sql);
        $results = array();
        
        foreach ($pages as $page) {
            $builder = $this->detect_page_builder($page->ID);
            $results[] = array(
                'id' => intval($page->ID),
                'title' => $page->post_title,
                'type' => $page->post_type,
                'item_type' => 'page',
                'builder' => $builder,
                'edit_url' => get_edit_post_link($page->ID, 'raw'),
                'view_url' => get_permalink($page->ID)
            );
        }
        
        return $results;
    }
    
    private function get_products($search = '') {
        global $wpdb;
        
        $where = "p.post_status = 'publish' AND p.post_type = 'product'";
        $params = array();
        
        if (!empty($search) && strlen($search) >= 2) {
            $like = '%' . $wpdb->esc_like($search) . '%';
            $where .= " AND (p.post_title LIKE %s OR p.post_name LIKE %s)";
            $params[] = $like;
            $params[] = $like;
        }
        
        $sql = "SELECT p.ID, p.post_title, p.post_type
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_mtp_source_product'
                WHERE {$where}
                  AND pm1.meta_id IS NULL
                ORDER BY p.post_title ASC
                LIMIT 200";
        
        if (!empty($params)) {
            $sql = $wpdb->prepare($sql, $params);
        }
        
        $products = $wpdb->get_results($sql);
        $results = array();
        
        foreach ($products as $product) {
            $wc_product = wc_get_product($product->ID);
            $product_type = $wc_product ? $wc_product->get_type() : 'simple';
            
            $results[] = array(
                'id' => intval($product->ID),
                'title' => $product->post_title,
                'type' => 'product',
                'item_type' => 'product',
                'product_type' => $product_type,
                'sku' => $wc_product ? $wc_product->get_sku() : '',
                'edit_url' => get_edit_post_link($product->ID, 'raw'),
                'view_url' => get_permalink($product->ID)
            );
        }
        
        return $results;
    }
    
    private function get_product_categories($search = '') {
        $args = array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'number' => 200
        );
        
        if (!empty($search) && strlen($search) >= 2) {
            $args['search'] = $search;
        }
        
        $terms = get_terms($args);
        $results = array();
        
        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $results[] = array(
                    'id' => intval($term->term_id),
                    'title' => $term->name,
                    'type' => 'product_cat',
                    'item_type' => 'product_cat',
                    'slug' => $term->slug,
                    'count' => $term->count,
                    'parent' => $term->parent,
                    'edit_url' => get_edit_term_link($term->term_id, 'product_cat'),
                    'view_url' => get_term_link($term)
                );
            }
        }
        
        return $results;
    }
    
    private function get_product_tags($search = '') {
        $args = array(
            'taxonomy' => 'product_tag',
            'hide_empty' => false,
            'number' => 200
        );
        
        if (!empty($search) && strlen($search) >= 2) {
            $args['search'] = $search;
        }
        
        $terms = get_terms($args);
        $results = array();
        
        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $results[] = array(
                    'id' => intval($term->term_id),
                    'title' => $term->name,
                    'type' => 'product_tag',
                    'item_type' => 'product_tag',
                    'slug' => $term->slug,
                    'count' => $term->count,
                    'edit_url' => get_edit_term_link($term->term_id, 'product_tag'),
                    'view_url' => get_term_link($term)
                );
            }
        }
        
        return $results;
    }
    
    private function get_product_attributes($search = '') {
        global $wpdb;
        
        $attributes = wc_get_attribute_taxonomies();
        $results = array();
        
        foreach ($attributes as $attribute) {
            $taxonomy = 'pa_' . $attribute->attribute_name;
            
            // Get attribute terms
            $terms = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false
            ));
            
            if (!is_wp_error($terms)) {
                foreach ($terms as $term) {
                    if (!empty($search) && stripos($term->name, $search) === false) {
                        continue;
                    }
                    
                    $results[] = array(
                        'id' => intval($term->term_id),
                        'title' => $term->name,
                        'type' => 'attribute',
                        'item_type' => 'attribute',
                        'taxonomy' => $taxonomy,
                        'attribute_label' => $attribute->attribute_label,
                        'slug' => $term->slug,
                        'edit_url' => get_edit_term_link($term->term_id, $taxonomy)
                    );
                }
            }
        }
        
        return $results;
    }
    
    // ==========================================
    // AJAX: CHECK BALANCE
    // ==========================================
    
    public function ajax_check_balance() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $target_lang = isset($_POST['target_lang']) ? sanitize_text_field($_POST['target_lang']) : null;
        
        $api_client = new MTP_API_Client();
        $result = $api_client->check_balance();
        
        if (!$result['success']) {
            wp_send_json_error($result['message']);
        }
        
        $tokens = intval($result['tokens_remaining']);
        $range = MTP_Word_Estimator::tokens_to_words_range($tokens, $target_lang);
        
        wp_send_json_success(array(
            'tokens_remaining' => $tokens,
            'words_min' => $range['min'],
            'words_max' => $range['max']
        ));
    }
    
    // ==========================================
    // CREATE TRANSLATED PAGE
    // ==========================================
    
    private function extract_elementor_text($data, $text = '') {
        if (!is_array($data)) return $text;
        
        $text_fields = array(
            // Basic widgets
            'title', 'description', 'editor', 'text', 'content',
            'heading_title', 'button_text', 'caption', 'html',
            // Testimonial
            'testimonial_content', 'testimonial_name', 'testimonial_job',
            // Alert, tabs, accordion
            'alert_title', 'alert_description', 'tab_title', 'tab_content',
            'accordion_title', 'accordion_content',
            // List, counter, etc.
            'item_description', 'title_text', 'description_text',
            'inner_text', 'link_text', 'list_text',
            // Price table
            'heading', 'sub_heading', 'period', 'footer_additional_info',
            'ribbon_title', 'item_text',
            // CTA, flip box
            'cta_text', 'flipbox_front_title', 'flipbox_front_description',
            'flipbox_back_title', 'flipbox_back_description',
            // Image box, icon box
            'title_text_a', 'title_text_b', 'description_text_a', 'description_text_b',
            // Blockquote, author
            'blockquote_content', 'tweet_text', 'author_name',
            // Form fields
            'field_label', 'placeholder', 'button_text_a',
            // Slides, carousel
            'slide_heading', 'slide_description', 'slide_button_text',
            // Nav menu
            'prefix_text', 'suffix_text',
            // Animated headline
            'before_text', 'highlighted_text', 'after_text', 'rotating_text',
            // Call to action
            'primary_cta_text', 'secondary_cta_text'
        );
        
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $text = $this->extract_elementor_text($value, $text);
            } elseif (is_string($value) && in_array($key, $text_fields)) {
                $clean = trim(strip_tags($value));
                if (!empty($clean)) {
                    $text .= ' ' . $clean;
                }
            }
        }
        
        return $text;
    }
    
    // ==========================================
    // ELEMENTOR SEGMENT-BASED TRANSLATION
    // ==========================================
    
    /**
     * Translate Elementor content by extracting text segments,
     * translating them as standard content, and rebuilding JSON.
     * This is more reliable than sending raw Elementor JSON to the API.
     * 
     * @param string $elementor_json Raw _elementor_data JSON string
     * @param string $title Page title
     * @param string $target_lang Target language code
     * @return array Result with translated_content, translated_title, etc.
     */
    private function translate_elementor_segments($elementor_json, $title, $target_lang) {
        // 1. Parse JSON
        $data = json_decode($elementor_json, true);
        if (!is_array($data) || empty($data)) {
            mtp_log('Elementor segments: Failed to parse JSON or empty data');
            return array('success' => false, 'message' => 'Invalid Elementor data');
        }
        
        // 2. Extract all translatable segments with their paths
        $segments = array();
        $this->collect_elementor_segments($data, $segments);
        
        mtp_log('Elementor segments: Found ' . count($segments) . ' translatable segments');
        
        if (empty($segments)) {
            // No text to translate - return original JSON as-is
            return array(
                'success' => true,
                'translated_content' => $elementor_json,
                'translated_title' => $title,
                'translated_slug' => '',
                'tokens_used' => 0
            );
        }
        
        // 3. Build combined HTML block with segment markers
        //    Each segment wrapped in <div id="mtp-N"> to preserve structure
        $html_block = '';
        foreach ($segments as $idx => $seg) {
            $html_block .= '<div id="mtp-' . $idx . '">' . $seg['value'] . '</div>' . "\n";
        }
        
        mtp_log('Elementor segments: Combined HTML length = ' . strlen($html_block));
        
        // 4. Send to API as standard content for translation
        $api_client = new MTP_API_Client();
        $result = $api_client->translate($html_block, $title, $target_lang, 'standard');
        
        if (!$result['success']) {
            mtp_log('Elementor segments: API translation failed - ' . $result['message']);
            return $result;
        }
        
        $translated_html = isset($result['translated_content']) ? $result['translated_content'] : '';
        mtp_log('Elementor segments: Received translated HTML length = ' . strlen($translated_html));
        
        // 5. Parse translated segments from response
        $translated_segments = $this->parse_elementor_translated($translated_html, count($segments));
        
        $matched = 0;
        foreach ($translated_segments as $v) {
            if ($v !== null) $matched++;
        }
        mtp_log('Elementor segments: Matched ' . $matched . '/' . count($segments) . ' segments');
        
        // 6. If primary parsing failed, try fallback methods
        if ($matched < count($segments) * 0.3) {
            mtp_log('Elementor segments: Low match rate, trying fallback parsing');
            $translated_segments = $this->parse_elementor_fallback($translated_html, $segments);
            
            $matched = 0;
            foreach ($translated_segments as $v) {
                if ($v !== null) $matched++;
            }
            mtp_log('Elementor segments: Fallback matched ' . $matched . '/' . count($segments));
        }
        
        // 7. Replace translated values in original Elementor data
        $replaced = 0;
        foreach ($segments as $idx => $seg) {
            if (isset($translated_segments[$idx]) && $translated_segments[$idx] !== null && $translated_segments[$idx] !== '') {
                $this->set_elementor_value($data, $seg['path'], $translated_segments[$idx]);
                $replaced++;
            }
        }
        
        mtp_log('Elementor segments: Replaced ' . $replaced . '/' . count($segments) . ' segments in JSON');
        
        // 8. Encode back to JSON
        $json_flags = JSON_UNESCAPED_UNICODE;
        if (defined('JSON_UNESCAPED_SLASHES')) {
            $json_flags = $json_flags | JSON_UNESCAPED_SLASHES;
        }
        $translated_json = json_encode($data, $json_flags);
        
        if ($translated_json === false) {
            mtp_log('Elementor segments: json_encode failed - ' . json_last_error_msg());
            return array('success' => false, 'message' => 'Failed to encode translated Elementor data');
        }
        
        return array(
            'success' => true,
            'translated_content' => $translated_json,
            'translated_title' => isset($result['translated_title']) ? $result['translated_title'] : $title,
            'translated_slug' => isset($result['translated_slug']) ? $result['translated_slug'] : '',
            'tokens_used' => isset($result['tokens_used']) ? intval($result['tokens_used']) : 0,
            'segments_total' => count($segments),
            'segments_translated' => $replaced
        );
    }
    
    /**
     * Recursively collect translatable text segments from Elementor JSON
     * Records path to each segment for later replacement
     * 
     * @param array $data Elementor data array
     * @param array &$segments Collected segments (by reference)
     * @param array $path Current path in the tree
     */
    private function collect_elementor_segments($data, &$segments, $path = array()) {
        if (!is_array($data)) return;
        
        $text_fields = array(
            'title', 'description', 'editor', 'text', 'content',
            'heading_title', 'button_text', 'caption', 'html',
            'testimonial_content', 'testimonial_name', 'testimonial_job',
            'alert_title', 'alert_description', 'tab_title', 'tab_content',
            'accordion_title', 'accordion_content',
            'item_description', 'title_text', 'description_text',
            'inner_text', 'link_text', 'list_text',
            'heading', 'sub_heading', 'period', 'footer_additional_info',
            'ribbon_title', 'item_text',
            'cta_text', 'flipbox_front_title', 'flipbox_front_description',
            'flipbox_back_title', 'flipbox_back_description',
            'title_text_a', 'title_text_b', 'description_text_a', 'description_text_b',
            'blockquote_content', 'tweet_text', 'author_name',
            'slide_heading', 'slide_description', 'slide_button_text',
            'before_text', 'highlighted_text', 'after_text', 'rotating_text',
            'primary_cta_text', 'secondary_cta_text'
        );
        
        foreach ($data as $key => $value) {
            $current_path = array_merge($path, array($key));
            
            if (is_array($value)) {
                $this->collect_elementor_segments($value, $segments, $current_path);
            } elseif (is_string($value) && in_array($key, $text_fields)) {
                // Filter: only include non-empty, meaningful text
                $clean = trim(strip_tags($value));
                if (!empty($clean) && strlen($clean) > 1) {
                    $segments[] = array(
                        'path'  => $current_path,
                        'value' => $value, // Keep original HTML
                        'key'   => $key
                    );
                }
            }
        }
    }
    
    /**
     * Parse translated segments from API response (primary method)
     * Looks for <div id="mtp-N"> markers
     * 
     * @param string $html Translated HTML from API
     * @param int $count Expected number of segments
     * @return array Indexed array of translated values (null if not found)
     */
    private function parse_elementor_translated($html, $count) {
        $translated = array();
        
        for ($i = 0; $i < $count; $i++) {
            $translated[$i] = null;
            
            // Primary: exact div id match
            $pattern = '/<div[^>]*id=["\']mtp-' . $i . '["\'][^>]*>([\s\S]*?)<\/div>/i';
            if (preg_match($pattern, $html, $match)) {
                $translated[$i] = trim($match[1]);
            }
        }
        
        return $translated;
    }
    
    /**
     * Fallback parser: split translated HTML by div tags and match by position
     * Used when primary parser fails (API modified the id attributes)
     * 
     * @param string $html Translated HTML
     * @param array $segments Original segments
     * @return array Indexed array of translated values
     */
    private function parse_elementor_fallback($html, $segments) {
        $translated = array();
        $count = count($segments);
        
        // Method 1: Try to split by any div tags
        $parts = preg_split('/<div[^>]*>/', $html);
        
        // Remove first empty element
        if (!empty($parts) && trim($parts[0]) === '') {
            array_shift($parts);
        }
        
        // Clean closing tags from each part
        $clean_parts = array();
        foreach ($parts as $part) {
            $cleaned = preg_replace('/<\/div>/', '', $part);
            $cleaned = trim($cleaned);
            if (!empty($cleaned)) {
                $clean_parts[] = $cleaned;
            }
        }
        
        // Map by position
        for ($i = 0; $i < $count; $i++) {
            $translated[$i] = isset($clean_parts[$i]) ? $clean_parts[$i] : null;
        }
        
        // Method 2: If still no matches, try splitting by double newlines
        $matched = 0;
        foreach ($translated as $v) {
            if ($v !== null) $matched++;
        }
        
        if ($matched < $count * 0.3) {
            // Strip all HTML tags and split by double newlines or <br> tags
            $plain = preg_replace('/<br\s*\/?>/i', "\n", $html);
            $plain = strip_tags($plain);
            $blocks = preg_split('/\n{2,}/', $plain, -1, PREG_SPLIT_NO_EMPTY);
            
            for ($i = 0; $i < $count; $i++) {
                if (isset($blocks[$i])) {
                    $translated[$i] = trim($blocks[$i]);
                }
            }
        }
        
        return $translated;
    }
    
    /**
     * Set a value at a specific path in the Elementor data tree
     * 
     * @param array &$data Elementor data (by reference)
     * @param array $path Path to the target value
     * @param mixed $value New value to set
     */
    private function set_elementor_value(&$data, $path, $value) {
        $ref = &$data;
        foreach ($path as $key) {
            if (!isset($ref[$key])) return;
            $ref = &$ref[$key];
        }
        $ref = $value;
    }
    
    private function create_translated_page($original_post, $translation_result, $target_lang, $link_lang = '', $content_type = 'standard', $as_draft = false) {
        $translated_title = !empty($translation_result['translated_title']) 
            ? $translation_result['translated_title'] 
            : $original_post->post_title;
        
        if (!empty($translation_result['translated_slug'])) {
            $translated_slug = $translation_result['translated_slug'];
        } else {
            $translated_slug = sanitize_title($translated_title);
        }
        
        if (empty($translated_slug)) {
            $translated_slug = $original_post->post_name . '-' . $target_lang;
        }
        
        $post_content = ($content_type === 'elementor') ? '' : $translation_result['translated_content'];
        
        $new_post_data = array(
            'post_title' => $translated_title,
            'post_name' => $translated_slug,
            'post_content' => $post_content,
            'post_status' => $as_draft ? 'draft' : 'publish',
            'post_type' => $original_post->post_type,
            'post_author' => get_current_user_id(),
            'post_parent' => $original_post->post_parent
        );
        
        $new_page_id = wp_insert_post($new_post_data);
        
        if (is_wp_error($new_page_id)) {
            return $new_page_id;
        }
        
        $this->copy_post_meta($original_post->ID, $new_page_id, $translation_result['translated_content'], $content_type);
        
        $thumbnail_id = get_post_thumbnail_id($original_post->ID);
        if ($thumbnail_id) {
            set_post_thumbnail($new_page_id, $thumbnail_id);
        }
        
        $taxonomies = get_object_taxonomies($original_post->post_type);
        foreach ($taxonomies as $taxonomy) {
            $terms = wp_get_object_terms($original_post->ID, $taxonomy, array('fields' => 'ids'));
            if (!is_wp_error($terms)) {
                wp_set_object_terms($new_page_id, $terms, $taxonomy);
            }
        }
        
        update_post_meta($new_page_id, '_mtp_source_page', $original_post->ID);
        update_post_meta($new_page_id, '_mtp_target_lang', $target_lang);
        update_post_meta($new_page_id, '_mtp_content_type', $content_type);
        update_post_meta($new_page_id, '_mtp_translated_date', current_time('mysql'));
        
        if ($content_type === 'elementor') {
            $this->regenerate_elementor_css($new_page_id);
        }
        
        if (!empty($link_lang)) {
            $this->link_translation_to_multilingual($original_post->ID, $new_page_id, $link_lang);
        }
        
        return $new_page_id;
    }
    
    private function regenerate_elementor_css($post_id) {
        if (!did_action('elementor/loaded')) {
            return;
        }
        
        if (class_exists('\Elementor\Core\Files\CSS\Post')) {
            $css_file = new \Elementor\Core\Files\CSS\Post($post_id);
            $css_file->update();
        }
        
        if (class_exists('\Elementor\Plugin')) {
            delete_post_meta($post_id, '_elementor_css');
            update_post_meta($post_id, '_elementor_element_cache', '');
        }
    }
    
    private function copy_post_meta($source_id, $target_id, $translated_content, $content_type) {
        $meta_keys = get_post_custom_keys($source_id);
        
        if (empty($meta_keys)) {
            return;
        }
        
        $exclude_keys = array('_edit_lock', '_edit_last', '_wp_old_slug', '_mtp_');
        
        foreach ($meta_keys as $meta_key) {
            $skip = false;
            foreach ($exclude_keys as $exclude) {
                if (strpos($meta_key, $exclude) === 0) {
                    $skip = true;
                    break;
                }
            }
            if ($skip) continue;
            
            if ($meta_key === '_elementor_data' && $content_type === 'elementor') {
                // Ensure content is a JSON string, not array
                $content_to_save = $translated_content;
                if (is_array($content_to_save)) {
                    $json_flags = JSON_UNESCAPED_UNICODE;
                    if (defined('JSON_UNESCAPED_SLASHES')) {
                        $json_flags = $json_flags | JSON_UNESCAPED_SLASHES;
                    }
                    $content_to_save = json_encode($content_to_save, $json_flags);
                }
                
                // Validate JSON before saving
                $test_decode = json_decode($content_to_save, true);
                if (json_last_error() !== JSON_ERROR_NONE || !is_array($test_decode)) {
                    mtp_log('copy_post_meta: WARNING - Invalid Elementor JSON, saving original');
                    // Save original elementor data as fallback
                    $original_data = get_post_meta($source_id, '_elementor_data', true);
                    if (!empty($original_data)) {
                        update_post_meta($target_id, '_elementor_data', wp_slash($original_data));
                    }
                    continue;
                }
                
                update_post_meta($target_id, '_elementor_data', wp_slash($content_to_save));
                
                // Verify saved correctly
                $saved = get_post_meta($target_id, '_elementor_data', true);
                if (empty($saved)) {
                    mtp_log('copy_post_meta: WARNING - Elementor data save verification failed');
                }
                
                continue;
            }
            
            if ($meta_key === '_aviaLayoutBuilderCleanData' && $content_type === 'avia') {
                update_post_meta($target_id, '_aviaLayoutBuilderCleanData', $translated_content);
                continue;
            }
            
            $meta_values = get_post_meta($source_id, $meta_key);
            foreach ($meta_values as $meta_value) {
                add_post_meta($target_id, $meta_key, $meta_value);
            }
        }
    }
    
    // ==========================================
    // MULTILINGUAL LINKING
    // ==========================================
    
    private function link_translation_to_multilingual($original_id, $translated_id, $target_lang) {
        // Polylang
        if (function_exists('pll_set_post_language') && function_exists('pll_save_post_translations')) {
            $source_lang = function_exists('pll_get_post_language') 
                ? pll_get_post_language($original_id) 
                : pll_default_language();
            
            pll_set_post_language($translated_id, $target_lang);
            
            $translations = function_exists('pll_get_post_translations') 
                ? pll_get_post_translations($original_id) 
                : array();
            
            if (!empty($source_lang)) {
                $translations[$source_lang] = $original_id;
            }
            $translations[$target_lang] = $translated_id;
            
            pll_save_post_translations($translations);
            
            update_post_meta($translated_id, '_mtp_multilingual_plugin', 'polylang');
            update_post_meta($translated_id, '_mtp_multilingual_linked', 'yes');
            return;
        }
        
        // WPML
        if (has_filter('wpml_element_type') && has_action('wpml_set_element_language_details')) {
            $post_type = get_post_type($original_id);
            $element_type = apply_filters('wpml_element_type', $post_type);
            $trid = apply_filters('wpml_element_trid', null, $original_id, $element_type);
            
            if (!empty($trid)) {
                $source_lang_details = apply_filters('wpml_element_language_details', null, array(
                    'element_id' => $original_id,
                    'element_type' => $element_type
                ));
                $source_lang = isset($source_lang_details->language_code) ? $source_lang_details->language_code : null;
                
                do_action('wpml_set_element_language_details', array(
                    'element_id' => $translated_id,
                    'element_type' => $element_type,
                    'trid' => $trid,
                    'language_code' => $target_lang,
                    'source_language_code' => $source_lang
                ));
                
                update_post_meta($translated_id, '_mtp_multilingual_plugin', 'wpml');
                update_post_meta($translated_id, '_mtp_multilingual_linked', 'yes');
            }
        }
    }
    
    public function detect_multilingual_plugin() {
        if (function_exists('pll_set_post_language')) return 'polylang';
        if (has_filter('wpml_element_type')) return 'wpml';
        return 'none';
    }
    
    public function get_multilingual_languages() {
        $languages = array();
        
        if (function_exists('pll_languages_list')) {
            $pll_langs = pll_languages_list(array('fields' => ''));
            if (!empty($pll_langs)) {
                foreach ($pll_langs as $lang) {
                    $languages[$lang->slug] = $lang->name;
                }
                return $languages;
            }
        }
        
        if (has_filter('wpml_active_languages')) {
            $wpml_langs = apply_filters('wpml_active_languages', null, array('skip_missing' => 0));
            if (!empty($wpml_langs)) {
                foreach ($wpml_langs as $code => $lang) {
                    $languages[$code] = $lang['native_name'];
                }
            }
        }
        
        return $languages;
    }
    
    // ==========================================
    // RENDER PAGES
    // ==========================================
    
    public function render_admin_page() {
        $client_id = get_option('mtp_client_id', '');
        
        if (empty($client_id)) {
            echo '<div class="wrap"><h1>MisterTranslate</h1>';
            echo '<div class="notice notice-warning" style="padding: 15px;">';
            echo '<p style="margin: 0 0 10px 0; font-size: 14px;"><strong>API credentials required</strong></p>';
            echo '<p style="margin: 0;">Please <a href="' . admin_url('admin.php?page=mistertranslate-settings') . '">configure your API credentials</a> to start translating.</p>';
            echo '<p style="margin: 10px 0 0 0;">Don\'t have an account yet? <a href="' . admin_url('admin.php?page=mistertranslate-settings#mtp-create-account') . '"><strong>Create one here &rarr;</strong></a></p>';
            echo '</div></div>';
            return;
        }
        
        include MTP_PLUGIN_DIR . 'templates/admin-page.php';
    }
    
    public function render_settings_page() {
        if (isset($_POST['mtp_save_settings']) && check_admin_referer('mtp_settings_nonce')) {
            $client_id = isset($_POST['mtp_client_id']) ? sanitize_text_field($_POST['mtp_client_id']) : '';
            $secret_key = isset($_POST['mtp_secret_key']) ? sanitize_text_field($_POST['mtp_secret_key']) : '';
            
            // Use security class to save encrypted credentials
            if (MTP_Security::save_credentials($client_id, $secret_key)) {
                echo '<div class="notice notice-success"><p>Settings saved securely!</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>Error saving settings. Please check your credentials format.</p></div>';
            }
        }
        
        // Save notification settings
        if (isset($_POST['mtp_save_notifications']) && check_admin_referer('mtp_settings_nonce')) {
            MTP_Notifications::save_settings(array(
                'low_balance_enabled' => isset($_POST['mtp_low_balance_enabled']),
                'low_balance_threshold' => isset($_POST['mtp_low_balance_threshold']) ? intval($_POST['mtp_low_balance_threshold']) : 1000,
                'notification_email' => isset($_POST['mtp_notification_email']) ? sanitize_email($_POST['mtp_notification_email']) : ''
            ));
            echo '<div class="notice notice-success"><p>Notification settings saved!</p></div>';
        }
        
        include MTP_PLUGIN_DIR . 'templates/settings-page.php';
    }
    
    // ==========================================
    // NEW v4.2 AJAX HANDLERS
    // ==========================================
    
    /**
     * AJAX: Estimate translation cost
     */
    public function ajax_estimate_cost() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $item_ids = isset($_POST['item_ids']) ? array_map('intval', (array) $_POST['item_ids']) : array();
        $item_type = isset($_POST['item_type']) ? sanitize_text_field($_POST['item_type']) : 'page';
        $target_langs = isset($_POST['target_langs']) ? array_map('sanitize_text_field', (array) $_POST['target_langs']) : array();
        
        if (empty($item_ids) || empty($target_langs)) {
            wp_send_json_error('Missing parameters');
        }
        
        $total_words = 0;
        
        foreach ($item_ids as $item_id) {
            if ($item_type === 'product' && $this->is_woocommerce_active()) {
                $product = wc_get_product($item_id);
                if ($product) {
                    $text = $product->get_name() . ' ' . $product->get_description() . ' ' . $product->get_short_description();
                    $total_words += MTP_Word_Estimator::count_words($text);
                }
            } elseif (in_array($item_type, array('product_cat', 'product_tag', 'attribute'))) {
                $term = get_term($item_id);
                if ($term && !is_wp_error($term)) {
                    $text = $term->name . ' ' . $term->description;
                    $total_words += MTP_Word_Estimator::count_words($text);
                }
            } else {
                $content_data = $this->get_page_content($item_id);
                if ($content_data['type'] === 'elementor') {
                    $data = json_decode($content_data['content'], true);
                    $text = $this->extract_elementor_text($data);
                } else {
                    $text = $content_data['content'];
                }
                $total_words += MTP_Word_Estimator::count_words($text);
            }
        }
        
        // Multiply by number of languages
        $total_words_all_langs = $total_words * count($target_langs);
        
        // Estimate tokens (average ratio)
        $tokens_min = intval($total_words_all_langs * 4.0);
        $tokens_max = intval($total_words_all_langs * 5.5);
        
        wp_send_json_success(array(
            'words_per_item' => $total_words,
            'total_words' => $total_words_all_langs,
            'tokens_min' => $tokens_min,
            'tokens_max' => $tokens_max,
            'languages_count' => count($target_langs),
            'items_count' => count($item_ids)
        ));
    }
    
    /**
     * AJAX: Preview content before translation
     */
    public function ajax_preview_content() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $item_id = isset($_POST['item_id']) ? intval($_POST['item_id']) : 0;
        $item_type = isset($_POST['item_type']) ? sanitize_text_field($_POST['item_type']) : 'page';
        
        if (!$item_id) {
            wp_send_json_error('Missing item ID');
        }
        
        $preview = array(
            'title' => '',
            'content' => '',
            'words' => 0,
            'type' => $item_type,
            'source_lang' => array()
        );
        
        if ($item_type === 'product' && $this->is_woocommerce_active()) {
            $product = wc_get_product($item_id);
            if ($product) {
                $preview['title'] = $product->get_name();
                $preview['content'] = wp_trim_words(strip_tags($product->get_description()), 100, '...');
                $full_text = $product->get_name() . ' ' . $product->get_description() . ' ' . $product->get_short_description();
                $preview['words'] = MTP_Word_Estimator::count_words($full_text);
                $preview['source_lang'] = MTP_Language_Detector::detect($full_text);
            }
        } elseif (in_array($item_type, array('product_cat', 'product_tag', 'attribute'))) {
            $term = get_term($item_id);
            if ($term && !is_wp_error($term)) {
                $preview['title'] = $term->name;
                $preview['content'] = wp_trim_words($term->description, 100, '...');
                $full_text = $term->name . ' ' . $term->description;
                $preview['words'] = MTP_Word_Estimator::count_words($full_text);
                $preview['source_lang'] = MTP_Language_Detector::detect($full_text);
            }
        } else {
            $post = get_post($item_id);
            if ($post) {
                $preview['title'] = $post->post_title;
                $content_data = $this->get_page_content($item_id);
                
                if ($content_data['type'] === 'elementor') {
                    $data = json_decode($content_data['content'], true);
                    $text = $this->extract_elementor_text($data);
                } else {
                    $text = $content_data['content'];
                }
                
                $preview['content'] = wp_trim_words(strip_tags($text), 100, '...');
                $preview['words'] = MTP_Word_Estimator::count_words($text);
                $preview['builder'] = $content_data['type'];
                $preview['source_lang'] = MTP_Language_Detector::detect($text);
            }
        }
        
        wp_send_json_success($preview);
    }
    
    /**
     * AJAX: Detect source language
     */
    public function ajax_detect_language() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $item_id = isset($_POST['item_id']) ? intval($_POST['item_id']) : 0;
        $item_type = isset($_POST['item_type']) ? sanitize_text_field($_POST['item_type']) : 'page';
        
        if (!$item_id) {
            wp_send_json_error('Missing item ID');
        }
        
        $text = '';
        
        if ($item_type === 'product' && $this->is_woocommerce_active()) {
            $product = wc_get_product($item_id);
            if ($product) {
                $text = $product->get_name() . ' ' . $product->get_description();
            }
        } elseif (in_array($item_type, array('product_cat', 'product_tag', 'attribute'))) {
            $term = get_term($item_id);
            if ($term && !is_wp_error($term)) {
                $text = $term->name . ' ' . $term->description;
            }
        } else {
            $content_data = $this->get_page_content($item_id);
            if ($content_data['type'] === 'elementor') {
                $data = json_decode($content_data['content'], true);
                $text = $this->extract_elementor_text($data);
            } else {
                $text = $content_data['content'];
            }
        }
        
        $detection = MTP_Language_Detector::detect($text);
        
        wp_send_json_success($detection);
    }
    
    /**
     * AJAX: Check for outdated translations
     */
    public function ajax_check_outdated() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $outdated = MTP_Translation_Tracker::find_outdated_translations(50);
        
        wp_send_json_success(array(
            'count' => count($outdated),
            'items' => $outdated
        ));
    }
    
    /**
     * AJAX: Re-translate existing translation
     */
    public function ajax_retranslate() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $translated_id = isset($_POST['translated_id']) ? intval($_POST['translated_id']) : 0;
        
        if (!$translated_id) {
            wp_send_json_error('Missing translated post ID');
        }
        
        // Get source info
        $source_id = get_post_meta($translated_id, '_mtp_source_page', true);
        $target_lang = get_post_meta($translated_id, '_mtp_target_lang', true);
        
        if (!$source_id || !$target_lang) {
            wp_send_json_error('Cannot determine source post or target language');
        }
        
        // Get source content
        $content_data = $this->get_page_content($source_id);
        $source_post = get_post($source_id);
        
        if (!$source_post) {
            wp_send_json_error('Source post not found');
        }
        
        // Check balance BEFORE translating
        $api_client = new MTP_API_Client();
        $balance = $api_client->check_balance();
        
        if (!$balance['success']) {
            wp_send_json_error('Could not verify balance: ' . $balance['message']);
        }
        
        // Count words and verify sufficient balance
        if ($content_data['type'] === 'elementor') {
            $temp_data = json_decode($content_data['content'], true);
            $temp_text = $this->extract_elementor_text($temp_data);
            $words_count = MTP_Word_Estimator::count_words($temp_text);
        } else {
            $words_count = MTP_Word_Estimator::count_words($content_data['content']);
        }
        
        $estimate = MTP_Word_Estimator::estimate_cost($words_count, $target_lang);
        $tokens_available = intval($balance['tokens_remaining']);
        $check = MTP_Word_Estimator::check_balance($tokens_available, $estimate);
        
        if (!$check['allowed']) {
            wp_send_json_error($check['message']);
        }
        
        // Translate - use segment-based for Elementor
        if ($content_data['type'] === 'elementor') {
            $result = $this->translate_elementor_segments($content_data['content'], $source_post->post_title, $target_lang);
        } else {
            $result = $api_client->translate($content_data['content'], $source_post->post_title, $target_lang, $content_data['type']);
        }
        
        if (!$result['success']) {
            wp_send_json_error($result['message']);
        }
        
        // Update translated post
        $update_data = array(
            'ID' => $translated_id,
            'post_title' => !empty($result['translated_title']) ? $result['translated_title'] : $source_post->post_title
        );
        
        if ($content_data['type'] === 'standard') {
            $update_data['post_content'] = $result['translated_content'];
        }
        
        wp_update_post($update_data);
        
        // Update page builder meta if needed
        if ($content_data['type'] === 'elementor') {
            // Ensure it's a valid JSON string
            $elementor_content = $result['translated_content'];
            if (is_array($elementor_content)) {
                $json_flags = JSON_UNESCAPED_UNICODE;
                if (defined('JSON_UNESCAPED_SLASHES')) {
                    $json_flags = $json_flags | JSON_UNESCAPED_SLASHES;
                }
                $elementor_content = json_encode($elementor_content, $json_flags);
            }
            update_post_meta($translated_id, '_elementor_data', wp_slash($elementor_content));
            
            // Regenerate Elementor CSS
            $this->regenerate_elementor_css($translated_id);
        } elseif ($content_data['type'] === 'avia') {
            update_post_meta($translated_id, '_aviaLayoutBuilderCleanData', $result['translated_content']);
        }
        
        // Update tracking hash
        MTP_Translation_Tracker::store_source_hash($translated_id, $source_id, $content_data['content']);
        update_post_meta($translated_id, '_mtp_translated_date', current_time('mysql'));
        
        // Get new balance
        $new_balance = $api_client->check_balance();
        $new_tokens = $new_balance['success'] ? intval($new_balance['tokens_remaining']) : 0;
        $new_range = MTP_Word_Estimator::tokens_to_words_range($new_tokens, $target_lang);
        
        // Check for low balance notification
        MTP_Notifications::check_low_balance($new_tokens);
        
        wp_send_json_success(array(
            'message' => 'Translation updated successfully',
            'tokens_used' => isset($result['tokens_used']) ? $result['tokens_used'] : 0,
            'new_balance' => array(
                'tokens' => $new_tokens,
                'words_min' => $new_range['min'],
                'words_max' => $new_range['max']
            )
        ));
    }
    
    /**
     * AJAX: Save glossary entry
     */
    public function ajax_save_glossary() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $action_type = isset($_POST['glossary_action']) ? sanitize_text_field($_POST['glossary_action']) : '';
        $term = isset($_POST['term']) ? sanitize_text_field($_POST['term']) : '';
        $translation = isset($_POST['translation']) ? sanitize_text_field($_POST['translation']) : '';
        $lang = isset($_POST['lang']) ? sanitize_text_field($_POST['lang']) : 'all';
        
        if (empty($term)) {
            wp_send_json_error('Term is required');
        }
        
        switch ($action_type) {
            case 'add_never':
                MTP_Glossary::add_never_translate($term);
                break;
            case 'remove_never':
                MTP_Glossary::remove_never_translate($term);
                break;
            case 'add_specific':
                if (empty($translation)) {
                    wp_send_json_error('Translation is required');
                }
                MTP_Glossary::add_specific_translation($term, $translation, $lang);
                break;
            case 'remove_specific':
                MTP_Glossary::remove_specific_translation($term, $lang);
                break;
            default:
                wp_send_json_error('Invalid action');
        }
        
        wp_send_json_success(array(
            'glossary' => MTP_Glossary::get_all(),
            'stats' => MTP_Glossary::get_stats()
        ));
    }
    
    /**
     * AJAX: Get glossary
     */
    public function ajax_get_glossary() {
        check_ajax_referer('mtp_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        wp_send_json_success(array(
            'glossary' => MTP_Glossary::get_all(),
            'stats' => MTP_Glossary::get_stats()
        ));
    }
    
    /**
     * Add translate quick action to posts/pages/products list
     */
    public function add_translate_row_action($actions, $post) {
        if (!current_user_can('manage_options')) {
            return $actions;
        }
        
        // Only for pages, posts and products
        $allowed_types = array('page', 'post', 'product');
        if (!in_array($post->post_type, $allowed_types)) {
            return $actions;
        }
        
        // Don't show for already translated posts
        if (get_post_meta($post->ID, '_mtp_source_page', true)) {
            return $actions;
        }
        
        $type = ($post->post_type === 'product') ? 'product' : $post->post_type;
        $url = admin_url('admin.php?page=mistertranslate&translate_id=' . $post->ID . '&type=' . $type);
        $actions['mtp_translate'] = '<a href="' . esc_url($url) . '" style="color: #ff8a4c;"><span class="dashicons dashicons-translation" style="font-size: 14px; vertical-align: middle;"></span> Translate</a>';
        
        return $actions;
    }
}

// Init
function mtp_init() {
    MisterTranslate::get_instance();
}
add_action('plugins_loaded', 'mtp_init');

// Activation
register_activation_hook(__FILE__, function() {
    if (get_option('mtp_client_id') === false) {
        add_option('mtp_client_id', '');
        add_option('mtp_secret_key', '');
    }
});
